// Initialize Telegram Web App
const tg = window.Telegram.WebApp;
tg.expand();
tg.ready();

// Get user data from Telegram
const telegramUser = tg.initDataUnsafe?.user;

// Check if running in Telegram
const isInTelegram = !!telegramUser;

// API Base URL
const API_URL = './api.php';

// Bot Configuration - Replace with your bot username
const BOT_USERNAME = 'bulb_coinbot';

// State
let userData = null;
let userFiles = [];

// Theme Functions
function initTheme() {
    const savedTheme = localStorage.getItem('theme') || 'light';
    if (savedTheme === 'dark') {
        document.body.classList.add('dark-mode');
        updateThemeIcon(true);
    }
}

function toggleTheme() {
    const isDark = document.body.classList.toggle('dark-mode');
    localStorage.setItem('theme', isDark ? 'dark' : 'light');
    updateThemeIcon(isDark);
}

function updateThemeIcon(isDark) {
    const icon = document.querySelector('#themeIcon');
    if (icon) {
        icon.className = isDark ? 'fas fa-sun' : 'fas fa-moon';
    }
}

// Modal Functions
function showModal() {
    document.getElementById('userNotLoadedModal').style.display = 'flex';
}

function closeModal() {
    document.getElementById('userNotLoadedModal').style.display = 'none';
}

// Initialize App
async function initApp() {
    try {
        // Show loading
        document.getElementById('loading').style.display = 'flex';
        
        // Check if user exists
        if (!isInTelegram) {
            // Show modal for non-Telegram access
            document.getElementById('loading').style.display = 'none';
            showModal();
            return;
        }
        
        initAppWithUser(telegramUser);
        
    } catch (error) {
        console.error('Error:', error);
        showError(error.message);
    }
}

// Initialize App with User
async function initAppWithUser(user) {
    try {
        document.getElementById('loading').style.display = 'flex';
        
        // Fetch user data
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getUserData',
                user_id: user.id
            })
        });

        if (!response.ok) {
            throw new Error('Failed to fetch user data');
        }

        const data = await response.json();
        
        console.log('API Response:', data); // Debug log
        
        if (data.success && data.data) {
            userData = data.data.user;
            userFiles = data.data.files || [];
            
            if (!userData) {
                throw new Error('اطلاعات کاربر یافت نشد');
            }
            
            renderUserInfo();
            renderWarnings();
            renderFiles();
            fetchBroadcastMessages(user.id);
            
            // Hide loading, show app
            document.getElementById('loading').style.display = 'none';
            document.getElementById('app').style.display = 'block';
            
            // Apply Telegram theme
            applyTelegramTheme();
        } else {
            throw new Error(data.message || 'خطا در دریافت اطلاعات');
        }
        
    } catch (error) {
        console.error('Error:', error);
        showError(error.message);
    }
}

// Apply Telegram Theme Colors
function applyTelegramTheme() {
    if (tg.themeParams) {
        const root = document.documentElement;
        
        if (tg.themeParams.bg_color) {
            document.body.style.background = tg.themeParams.bg_color;
        }
        
        if (tg.themeParams.text_color) {
            root.style.setProperty('--text-color', tg.themeParams.text_color);
        }
    }
}

// Render User Info
function renderUserInfo() {
    // User name and avatar
    const userName = userData.first_name + (userData.last_name ? ' ' + userData.last_name : '');
    document.getElementById('userName').textContent = userName;
    
    // Username
    if (userData.username) {
        document.getElementById('userUsername').textContent = '@' + userData.username;
    } else {
        document.getElementById('userUsername').textContent = 'بدون نام کاربری';
    }
    
    // Avatar
    const avatarText = userName.substring(0, 2).toUpperCase();
    document.getElementById('avatarText').textContent = avatarText;
    
    // Stats
    document.getElementById('downloadCount').textContent = userData.download_count || 0;
    document.getElementById('warningCount').textContent = userData.warning_count || 0;
    
    // Status
    const statusStat = document.getElementById('statusStat');
    const statusIcon = document.getElementById('statusIcon');
    const statusText = document.getElementById('statusText');
    
    if (userData.is_blocked) {
        statusStat.classList.add('blocked');
        statusIcon.className = 'fas fa-ban stat-icon';
        statusText.textContent = 'مسدود';
        
        // Show blocked message
        document.getElementById('blockedMessage').style.display = 'block';
        
        // Show appeal status if exists
        if (userData.appeal) {
            renderAppealStatus(userData.appeal);
        }
    } else {
        statusIcon.className = 'fas fa-check-circle stat-icon';
        statusText.textContent = 'فعال';
    }
    
    // Warning color
    const warningCount = userData.warning_count || 0;
    if (warningCount >= 3) {
        document.getElementById('warningCount').style.color = '#dc2626';
    } else if (warningCount >= 2) {
        document.getElementById('warningCount').style.color = '#f59e0b';
    }
}

// Render Warnings
function renderWarnings() {
    const warnings = userData.warnings || [];
    
    if (warnings.length === 0) {
        return;
    }
    
    const warningsSection = document.getElementById('warningsSection');
    const warningsList = document.getElementById('warningsList');
    
    warningsSection.style.display = 'block';
    warningsList.innerHTML = '';
    
    warnings.forEach((warning, index) => {
        const warningItem = document.createElement('div');
        warningItem.className = 'warning-item';
        
        const date = new Date(warning.date);
        const formattedDate = date.toLocaleDateString('fa-IR');
        
        warningItem.innerHTML = `
            <div class="warning-reason"><i class="fas fa-exclamation-circle"></i> ${escapeHtml(warning.reason)}</div>
            <div class="warning-date"><i class="far fa-calendar"></i> ${formattedDate}</div>
            <div class="warning-admin"><i class="fas fa-user-shield"></i> توسط ادمین</div>
        `;
        
        warningsList.appendChild(warningItem);
    });
}

// Render Files
function renderFiles() {
    console.log('renderFiles called with', userFiles.length, 'files');
    const filesList = document.getElementById('filesList');
    const emptyState = document.getElementById('emptyState');
    const fileCount = document.getElementById('fileCount');
    
    if (userFiles.length === 0) {
        emptyState.style.display = 'block';
        filesList.style.display = 'none';
        fileCount.textContent = '0 فایل';
        return;
    }
    
    emptyState.style.display = 'none';
    filesList.style.display = 'flex';
    fileCount.textContent = `${userFiles.length} فایل`;
    
    filesList.innerHTML = '';
    
    userFiles.forEach(file => {
        const fileItem = createFileItem(file);
        filesList.appendChild(fileItem);
        console.log('File item added:', file.file_name);
    });
}

// Create File Item
function createFileItem(file) {
    const fileItem = document.createElement('div');
    fileItem.className = 'file-item';
    
    // File icon based on type
    const icon = getFileIcon(file.file_type);
    
    // File size
    const size = formatFileSize(file.file_size);
    
    // Download date
    const date = new Date(file.upload_date);
    const formattedDate = date.toLocaleDateString('fa-IR', { 
        year: 'numeric', 
        month: 'short', 
        day: 'numeric' 
    });
    
    // Rating
    const rating = file.average_rating ? `<i class="fas fa-star" style="color: #FF9F0A;"></i> ${file.average_rating.toFixed(1)}` : '<i class="far fa-star" style="color: #8E8E93;"></i> --';
    
    // File title
    const title = file.title || file.file_name;
    
    fileItem.innerHTML = `
        <div class="file-icon">${icon}</div>
        <div class="file-info">
            <div class="file-title">${escapeHtml(title)}</div>
            <div class="file-meta">
                <span class="file-size"><i class="fas fa-hdd"></i> ${size}</span>
                <span class="file-date"><i class="far fa-calendar"></i> ${formattedDate}</span>
                <span class="file-rating">${rating}</span>
            </div>
        </div>
    `;
    
    // Click handler - استفاده از bind برای اطمینان از کار کردن
    fileItem.style.cursor = 'pointer';
    fileItem.onclick = function() {
        console.log('File clicked:', file.file_name);
        showFileDetails(file);
    };
    
    return fileItem;
}

// Show File Details Modal
let selectedFile = null;

function showFileDetails(file) {
    console.log('showFileDetails called with:', file);
    selectedFile = file;
    const modal = document.getElementById('fileDetailsModal');
    const title = document.getElementById('fileDetailsTitle');
    const body = document.getElementById('fileDetailsBody');
    
    if (!modal) {
        console.error('Modal not found!');
        return;
    }
    
    title.textContent = file.title || file.file_name;
    
    const date = new Date(file.upload_date);
    const formattedDate = date.toLocaleDateString('fa-IR', {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
    
    body.innerHTML = `
        <div class="file-detail-item">
            <i class="fas fa-file"></i>
            <div class="file-detail-info">
                <div class="file-detail-label">نام فایل</div>
                <div class="file-detail-value">${escapeHtml(file.file_name)}</div>
            </div>
        </div>
        <div class="file-detail-item">
            <i class="fas fa-hdd"></i>
            <div class="file-detail-info">
                <div class="file-detail-label">حجم</div>
                <div class="file-detail-value">${formatFileSize(file.file_size)}</div>
            </div>
        </div>
        <div class="file-detail-item">
            <i class="far fa-calendar"></i>
            <div class="file-detail-info">
                <div class="file-detail-label">تاریخ آپلود</div>
                <div class="file-detail-value">${formattedDate}</div>
            </div>
        </div>
        <div class="file-detail-item">
            <i class="fas fa-star"></i>
            <div class="file-detail-info">
                <div class="file-detail-label">امتیاز</div>
                <div class="file-detail-value">${file.average_rating ? file.average_rating.toFixed(1) : '--'} (${file.total_ratings || 0} رای)</div>
            </div>
        </div>
        <div class="file-detail-item">
            <i class="fas fa-tag"></i>
            <div class="file-detail-info">
                <div class="file-detail-label">نوع فایل</div>
                <div class="file-detail-value">${getFileTypeName(file.file_type)}</div>
            </div>
        </div>
    `;
    
    modal.style.display = 'flex';
}

function closeFileDetails() {
    document.getElementById('fileDetailsModal').style.display = 'none';
    selectedFile = null;
}

// Close modals on outside click
document.addEventListener('click', function(e) {
    const fileModal = document.getElementById('fileDetailsModal');
    const appealModal = document.getElementById('appealModal');
    
    if (e.target === fileModal) {
        closeFileDetails();
    }
    
    if (e.target === appealModal) {
        closeAppealModal();
    }
});

function downloadFile() {
    if (!selectedFile) return;
    
    // Open Telegram bot to download
    const fileId = selectedFile.id;
    const deepLink = `https://t.me/${BOT_USERNAME}?start=download_${fileId}`;
    
    // Try to use Telegram WebApp method first, fallback to window.open
    if (tg.openTelegramLink) {
        tg.openTelegramLink(deepLink);
    } else {
        window.open(deepLink, '_blank');
    }
    
    closeFileDetails();
    showNotification('درخواست دانلود ارسال شد. لطفاً ربات را باز کنید.', 'success');
}

function getFileTypeName(type) {
    const types = {
        'document': 'سند',
        'video': 'ویدیو',
        'audio': 'صوت',
        'photo': 'تصویر',
        'voice': 'صدا',
        'archive': 'آرشیو',
        'application': 'برنامه'
    };
    return types[type] || 'فایل';
}

// Show Notification
function showNotification(message, type = 'success') {
    const toast = document.getElementById('notificationToast');
    const icon = toast.querySelector('i');
    const text = toast.querySelector('.notification-text');
    
    // Update content
    text.textContent = message;
    
    // Update icon and class
    toast.className = 'notification-toast ' + type;
    if (type === 'success') {
        icon.className = 'fas fa-check-circle';
    } else if (type === 'error') {
        icon.className = 'fas fa-times-circle';
    }
    
    // Show toast
    setTimeout(() => {
        toast.classList.add('show');
    }, 100);
    
    // Hide after 3 seconds
    setTimeout(() => {
        toast.classList.remove('show');
    }, 3500);
}

// Get File Icon
function getFileIcon(type) {
    const icons = {
        'document': '<i class="fas fa-file-alt"></i>',
        'video': '<i class="fas fa-video"></i>',
        'audio': '<i class="fas fa-music"></i>',
        'photo': '<i class="fas fa-image"></i>',
        'voice': '<i class="fas fa-microphone"></i>',
        'archive': '<i class="fas fa-file-archive"></i>',
        'application': '<i class="fas fa-box"></i>'
    };
    
    return icons[type] || '<i class="fas fa-file"></i>';
}

// Format File Size
function formatFileSize(bytes) {
    if (bytes === 0) return '0 B';
    
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// Show Error
function showError(message) {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('errorMessage').style.display = 'block';
    document.getElementById('errorText').textContent = message;
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        initTheme();
        initApp();
    });
} else {
    initTheme();
    initApp();
}

// Render Appeal Status
function renderAppealStatus(appeal) {
    const appealStatusDiv = document.getElementById('appealStatus');
    
    // Don't show if no appeal exists or no status
    if (!appeal || !appeal.status || !appealStatusDiv) {
        if (appealStatusDiv) appealStatusDiv.style.display = 'none';
        return;
    }
    
    const status = appeal.status; // pending, accepted, rejected
    let icon, title, message;
    
    if (status === 'pending') {
        icon = 'fas fa-clock';
        title = 'اعتراض شما در حال بررسی است';
        message = 'اعتراض شما ثبت شده و به زودی توسط مدیران بررسی خواهد شد.';
    } else if (status === 'accepted') {
        icon = 'fas fa-check-circle';
        title = 'اعتراض شما پذیرفته شد';
        message = 'مسدودیت شما رفع شد. می‌توانید از ربات استفاده کنید.';
    } else if (status === 'rejected') {
        icon = 'fas fa-times-circle';
        title = 'اعتراض شما رد شد';
        message = 'متأسفانه اعتراض شما پذیرفته نشد.';
    }
    
    let html = `
        <div class="appeal-status-header">
            <i class="${icon}"></i>
            <h4>${title}</h4>
        </div>
        <div class="appeal-status-body">
            <p>${message}</p>
    `;
    
    // Show reject reason if exists
    if (status === 'rejected' && appeal.reject_reason) {
        html += `
            <div class="appeal-reject-reason">
                <strong>دلیل رد:</strong>
                ${escapeHtml(appeal.reject_reason)}
            </div>
        `;
    }
    
    // Show can retry message for rejected appeals
    if (status === 'rejected') {
        html += `
            <div class="appeal-can-retry">
                <i class="fas fa-redo"></i>
                شما می‌توانید مجدداً اعتراض کنید
            </div>
        `;
    }
    
    html += '</div>';
    
    appealStatusDiv.className = `appeal-status ${status}`;
    appealStatusDiv.innerHTML = html;
    appealStatusDiv.style.display = 'block';
}

// Appeal Accordion Functions
function toggleAppealForm() {
    const form = document.getElementById('appealFormAccordion');
    const success = document.getElementById('appealSuccessAccordion');
    const textarea = document.getElementById('appealTextAccordion');
    const toggleBtn = document.querySelector('.appeal-toggle-btn');
    const appealStatus = document.getElementById('appealStatus');
    
    if (form.style.display === 'none' || !form.style.display) {
        form.style.display = 'block';
        success.style.display = 'none';
        textarea.value = '';
        if (toggleBtn) toggleBtn.style.display = 'none';
        // Hide appeal status when opening form
        if (appealStatus) appealStatus.style.display = 'none';
        setTimeout(() => textarea.focus(), 100);
    } else {
        form.style.display = 'none';
        if (toggleBtn) toggleBtn.style.display = 'inline-flex';
        // Show appeal status again if it exists
        if (appealStatus && appealStatus.innerHTML) {
            appealStatus.style.display = 'block';
        }
    }
}

async function submitAppealAccordion() {
    const textarea = document.getElementById('appealTextAccordion');
    const appealText = textarea.value.trim();
    
    if (!appealText) {
        showNotification('لطفاً متن اعتراض را وارد کنید', 'error');
        return;
    }
    
    const submitBtn = document.querySelector('.appeal-form-accordion .submit-btn');
    const originalText = submitBtn.innerHTML;
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال ارسال...';
    
    try {
        const user = window.Telegram?.WebApp?.initDataUnsafe?.user || userData || {
            id: 123456789,
            first_name: 'کاربر',
            last_name: 'نمونه',
            username: 'sample_user'
        };
        
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'submitAppeal',
                user_id: user.id || user.user_id,
                appeal_text: appealText,
                user_info: {
                    first_name: user.first_name,
                    last_name: user.last_name || '',
                    username: user.username || ''
                }
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            document.querySelector('.appeal-form-body').style.display = 'none';
            document.getElementById('appealSuccessAccordion').style.display = 'block';
            // Hide old appeal status when new appeal is submitted
            const appealStatus = document.getElementById('appealStatus');
            if (appealStatus) appealStatus.style.display = 'none';
            showNotification('✅ اعتراض شما با موفقیت ثبت شد', 'success');
            
            setTimeout(() => {
                toggleAppealForm();
                document.querySelector('.appeal-form-body').style.display = 'block';
                document.querySelector('.appeal-toggle-btn').style.display = 'inline-flex';
            }, 3000);
        } else {
            showNotification('❌ خطا در ثبت اعتراض: ' + (result.message || 'خطای نامشخص'), 'error');
        }
    } catch (error) {
        console.error('Appeal Error:', error);
        showNotification('❌ خطا در ارسال اعتراض. لطفاً دوباره تلاش کنید.', 'error');
    } finally {
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalText;
    }
}

// Appeal Functions
function showAppealModal() {
    document.getElementById('appealModal').style.display = 'flex';
    document.getElementById('appealText').value = '';
    document.getElementById('appealText').focus();
}

function closeAppealModal() {
    document.getElementById('appealModal').style.display = 'none';
}

async function submitAppeal() {
    const appealText = document.getElementById('appealText').value.trim();
    
    if (!appealText) {
        tg.showAlert('لطفاً دلیل اعتراض خود را بنویسید');
        return;
    }
    
    if (appealText.length < 10) {
        tg.showAlert('لطفاً توضیحات بیشتری بنویسید (حداقل 10 حرف)');
        return;
    }
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'submitAppeal',
                user_id: userData.user_id,
                appeal_text: appealText,
                user_info: {
                    first_name: userData.first_name,
                    last_name: userData.last_name,
                    username: userData.username
                }
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            closeAppealModal();
            showNotification('✅ اعتراض شما با موفقیت ثبت شد و به زودی بررسی خواهد شد', 'success');
        } else {
            showNotification('❌ خطا در ثبت اعتراض: ' + (result.message || 'خطای نامشخص'), 'error');
        }
    } catch (error) {
        console.error('Appeal Error:', error);
        showNotification('❌ خطا در ارسال اعتراض. لطفاً دوباره تلاش کنید.', 'error');
    }
}

// Fetch and Render Broadcast Messages
async function fetchBroadcastMessages(userId) {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getBroadcastMessages',
                user_id: userId
            })
        });

        if (!response.ok) {
            throw new Error('Failed to fetch messages');
        }

        const data = await response.json();
        
        if (data.success && data.data && data.data.broadcasts) {
            renderBroadcastMessages(data.data.broadcasts);
        }
    } catch (error) {
        console.error('Error fetching broadcasts:', error);
    }
}

// Render Broadcast Messages
function renderBroadcastMessages(broadcasts) {
    const messagesList = document.getElementById('messagesList');
    const messagesEmptyState = document.getElementById('messagesEmptyState');
    const messageCount = document.getElementById('messageCount');
    
    if (!broadcasts || broadcasts.length === 0) {
        messagesEmptyState.style.display = 'block';
        messagesList.style.display = 'none';
        messageCount.textContent = '0 پیام';
        return;
    }
    
    messagesEmptyState.style.display = 'none';
    messagesList.style.display = 'block';
    messageCount.textContent = `${broadcasts.length} پیام`;
    
    messagesList.innerHTML = broadcasts.map(msg => `
        <div class="message-card">
            <div class="message-header">
                <div class="message-icon">
                    <i class="fas fa-bullhorn"></i>
                </div>
                <div class="message-meta">
                    <div class="message-label">پیام همگانی</div>
                    <div class="message-date">${msg.formatted_date || msg.sent_at}</div>
                </div>
            </div>
            <div class="message-body">${escapeHtml(msg.message)}</div>
            <div class="message-footer">
                <i class="fas fa-paper-plane"></i>
                <span>ارسال شده توسط مدیر</span>
            </div>
        </div>
    `).join('');
}

// Escape HTML to prevent XSS
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
