<?php
/**
 * مدیریت دیتابیس JSON
 * JSON Database Manager
 */

class Database {
    
    /**
     * خواندن داده از فایل JSON
     */
    public static function read($file) {
        if (!file_exists($file)) {
            self::write($file, []);
            return [];
        }
        
        $data = file_get_contents($file);
        return json_decode($data, true) ?: [];
    }
    
    /**
     * نوشتن داده در فایل JSON
     */
    public static function write($file, $data) {
        $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        return file_put_contents($file, $json);
    }
    
    /**
     * اضافه کردن فایل جدید
     */
    public static function addFile($fileId, $fileName, $fileType, $fileSize, $uploadedBy, $options = []) {
        $files = self::read(FILES_DB);
        
        $uniqueId = self::generateUniqueId();
        
        $files[$uniqueId] = [
            'id' => $uniqueId,
            'file_id' => $fileId,
            'file_name' => $fileName,
            'file_type' => $fileType,
            'file_size' => $fileSize,
            'uploaded_by' => $uploadedBy,
            'upload_date' => date('Y-m-d H:i:s'),
            'download_count' => 0,
            // گزینه‌های جدید
            'title' => $options['title'] ?? '',
            'caption' => $options['caption'] ?? '',
            'download_limit' => $options['download_limit'] ?? 0, // 0 = نامحدود
            'force_join_enabled' => $options['force_join_enabled'] ?? true,
            'expire_date' => $options['expire_date'] ?? null,
            'password' => $options['password'] ?? '', // رمز عبور فایل
            'is_active' => true,
            // سیستم امتیازدهی
            'ratings' => [], // آرایه [user_id => rating]
            'average_rating' => 0, // میانگین امتیاز
            'total_ratings' => 0 // تعداد امتیازدهندگان
        ];
        
        self::write(FILES_DB, $files);
        
        return $uniqueId;
    }
    
    /**
     * دریافت اطلاعات فایل
     */
    public static function getFile($uniqueId) {
        $files = self::read(FILES_DB);
        return isset($files[$uniqueId]) ? $files[$uniqueId] : null;
    }
    
    /**
     * افزایش تعداد دانلود
     */
    public static function incrementDownload($uniqueId) {
        $files = self::read(FILES_DB);
        
        if (isset($files[$uniqueId])) {
            $files[$uniqueId]['download_count']++;
            
            // بررسی محدودیت دانلود
            if ($files[$uniqueId]['download_limit'] > 0 && 
                $files[$uniqueId]['download_count'] >= $files[$uniqueId]['download_limit']) {
                $files[$uniqueId]['is_active'] = false;
            }
            
            self::write(FILES_DB, $files);
        }
    }
    
    /**
     * بروزرسانی فایل
     */
    public static function updateFile($uniqueId, $updates) {
        $files = self::read(FILES_DB);
        
        if (isset($files[$uniqueId])) {
            foreach ($updates as $key => $value) {
                $files[$uniqueId][$key] = $value;
            }
            self::write(FILES_DB, $files);
            return true;
        }
        
        return false;
    }
    
    /**
     * مدیریت وضعیت آپلود موقت
     */
    public static function setUploadState($userId, $state, $data = []) {
        $states = self::read(UPLOAD_TEMP_DB);
        
        $states[$userId] = [
            'state' => $state,
            'data' => $data,
            'timestamp' => time()
        ];
        
        self::write(UPLOAD_TEMP_DB, $states);
    }
    
    public static function getUploadState($userId) {
        $states = self::read(UPLOAD_TEMP_DB);
        return isset($states[$userId]) ? $states[$userId] : null;
    }
    
    public static function clearUploadState($userId) {
        $states = self::read(UPLOAD_TEMP_DB);
        
        if (isset($states[$userId])) {
            unset($states[$userId]);
            self::write(UPLOAD_TEMP_DB, $states);
        }
    }
    
    /**
     * حذف فایل
     */
    public static function deleteFile($uniqueId) {
        $files = self::read(FILES_DB);
        
        if (isset($files[$uniqueId])) {
            unset($files[$uniqueId]);
            self::write(FILES_DB, $files);
            return true;
        }
        
        return false;
    }
    
    /**
     * دریافت تمام فایل‌های آپلود شده
     */
    public static function getAllFiles() {
        return self::read(FILES_DB);
    }
    
    /**
     * ثبت کاربر
     */
    public static function addUser($userId, $username, $firstName, $lastName) {
        $users = self::read(USERS_DB);
        
        if (!isset($users[$userId])) {
            $users[$userId] = [
                'user_id' => $userId,
                'username' => $username,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'join_date' => date('Y-m-d H:i:s'),
                'download_count' => 0,
                'total_download_size' => 0,
                'is_blocked' => false,
                'downloaded_files' => [], // آرایه ID فایل‌ها
                'warnings' => [], // آرایه اخطارها
                'warning_count' => 0 // تعداد اخطارهای فعال
            ];
            
            self::write(USERS_DB, $users);
        }
    }
    
    /**
     * ثبت دانلود فایل برای کاربر
     */
    public static function addDownloadedFile($userId, $fileId) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            if (!isset($users[$userId]['downloaded_files'])) {
                $users[$userId]['downloaded_files'] = [];
            }
            
            // اضافه فایل به لیست اگر قبلاً دانلود نشده
            if (!in_array($fileId, $users[$userId]['downloaded_files'])) {
                $users[$userId]['downloaded_files'][] = $fileId;
                self::write(USERS_DB, $users);
            }
        }
    }
    
    /**
     * دریافت فایل‌های دانلود شده کاربر
     */
    public static function getUserDownloadedFiles($userId) {
        $users = self::read(USERS_DB);
        $user = isset($users[$userId]) ? $users[$userId] : null;
        
        if (!$user || empty($user['downloaded_files'])) {
            return [];
        }
        
        $files = self::getAllFiles();
        $downloadedFiles = [];
        
        foreach ($user['downloaded_files'] as $fileId) {
            if (isset($files[$fileId])) {
                $downloadedFiles[] = $files[$fileId];
            }
        }
        
        return $downloadedFiles;
    }
    
    /**
     * دریافت اطلاعات کاربر
     */
    public static function getUser($userId) {
        $users = self::read(USERS_DB);
        return isset($users[$userId]) ? $users[$userId] : null;
    }
    
    /**
     * دریافت لیست تمام کاربران
     */
    public static function getAllUsers() {
        $users = self::read(USERS_DB);
        return array_values($users);
    }
    
    /**
     * جستجوی کاربر با ID یا username
     */
    public static function searchUser($query) {
        $users = self::read(USERS_DB);
        
        // جستجو با ID عددی
        if (is_numeric($query) && isset($users[$query])) {
            return $users[$query];
        }
        
        // جستجو با username
        $query = ltrim($query, '@');
        foreach ($users as $user) {
            if (strtolower($user['username']) === strtolower($query)) {
                return $user;
            }
        }
        
        return null;
    }
    
    /**
     * بلاک کردن کاربر
     */
    public static function blockUser($userId) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            $users[$userId]['is_blocked'] = true;
            self::write(USERS_DB, $users);
            return true;
        }
        
        return false;
    }
    
    /**
     * آنبلاک کردن کاربر
     */
    public static function unblockUser($userId) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            $users[$userId]['is_blocked'] = false;
            self::write(USERS_DB, $users);
            return true;
        }
        
        return false;
    }
    
    /**
     * اضافه کردن اخطار به کاربر
     */
    public static function addWarning($userId, $reason, $adminId) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            if (!isset($users[$userId]['warnings'])) {
                $users[$userId]['warnings'] = [];
            }
            if (!isset($users[$userId]['warning_count'])) {
                $users[$userId]['warning_count'] = 0;
            }
            
            $warning = [
                'reason' => $reason,
                'admin_id' => $adminId,
                'date' => date('Y-m-d H:i:s'),
                'timestamp' => time()
            ];
            
            $users[$userId]['warnings'][] = $warning;
            $users[$userId]['warning_count']++;
            
            self::write(USERS_DB, $users);
            return true;
        }
        
        return false;
    }
    
    /**
     * دریافت تعداد اخطارهای کاربر
     */
    public static function getUserWarningCount($userId) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            return $users[$userId]['warning_count'] ?? 0;
        }
        
        return 0;
    }
    
    /**
     * دریافت لیست اخطارهای کاربر
     */
    public static function getUserWarnings($userId) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            return $users[$userId]['warnings'] ?? [];
        }
        
        return [];
    }
    
    /**
     * پاک کردن همه اخطارهای کاربر
     */
    public static function clearUserWarnings($userId) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            $users[$userId]['warnings'] = [];
            $users[$userId]['warning_count'] = 0;
            
            self::write(USERS_DB, $users);
            return true;
        }
        
        return false;
    }
    
    /**
     * حذف یک اخطار خاص
     */
    public static function removeWarning($userId, $warningIndex) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId]) && isset($users[$userId]['warnings'][$warningIndex])) {
            unset($users[$userId]['warnings'][$warningIndex]);
            $users[$userId]['warnings'] = array_values($users[$userId]['warnings']); // reindex
            $users[$userId]['warning_count'] = count($users[$userId]['warnings']);
            
            self::write(USERS_DB, $users);
            return true;
        }
        
        return false;
    }
    
    /**
     * بررسی بلاک بودن کاربر
     */
    public static function isUserBlocked($userId) {
        $users = self::read(USERS_DB);
        return isset($users[$userId]) && $users[$userId]['is_blocked'];
    }
    
    /**
     * دریافت آمار دانلود کاربر
     */
    public static function getUserDownloadStats($userId) {
        $files = self::getAllFiles();
        $downloadCount = 0;
        $totalSize = 0;
        $downloadedFiles = [];
        
        // شمارش دانلودهای این کاربر از طریق فایل‌ها
        // در اینجا فقط download_count کلی کاربر را از users.json برمی‌گردانیم
        $users = self::read(USERS_DB);
        $user = isset($users[$userId]) ? $users[$userId] : null;
        
        if (!$user) {
            return [
                'download_count' => 0,
                'total_size' => 0
            ];
        }
        
        return [
            'download_count' => $user['download_count'] ?? 0,
            'total_size' => $user['total_download_size'] ?? 0
        ];
    }
    
    /**
     * ثبت دانلود برای کاربر (برای آمار)
     */
    public static function recordUserDownload($userId, $fileSize) {
        $users = self::read(USERS_DB);
        
        if (isset($users[$userId])) {
            $users[$userId]['download_count'] = ($users[$userId]['download_count'] ?? 0) + 1;
            $users[$userId]['total_download_size'] = ($users[$userId]['total_download_size'] ?? 0) + $fileSize;
            self::write(USERS_DB, $users);
        }
    }
    
    /**
     * تولید آیدی یونیک
     */
    private static function generateUniqueId() {
        return bin2hex(random_bytes(8));
    }
    
    /**
     * دریافت تنظیمات
     */
    public static function getSettings() {
        $settings = self::read(DATA_PATH . 'settings.json');
        
        // تنظیمات پیش‌فرض
        $defaults = [
            'maintenance_mode' => false,
            'force_join' => false,
            'force_join_channels' => [], // آرایه کانال‌ها
            'welcome_message' => 'به ربات آپلودر خوش آمدید!',
            // تنظیمات اطلاع‌رسانی
            'notification_enabled' => false,
            'notification_channel' => '', // @channel یا ID کانال
            'notification_to_users' => false, // ارسال به کاربران
            'notification_to_channel' => false, // ارسال به کانال
            // تنظیمات دکمه‌های کاربران
            'user_button_info' => true, // دکمه اطلاعات من
            'user_button_downloads' => true, // دکمه فایل‌های دانلود شده
            'user_button_toprated' => true, // دکمه بهترین فایل‌ها
            // لیست ادمین‌ها
            'admins_list' => [] // آرایه از user_id ادمین‌ها
        ];
        
        return array_merge($defaults, $settings);
    }
    
    /**
     * ذخیره تنظیمات
     */
    public static function saveSettings($settings) {
        return self::write(DATA_PATH . 'settings.json', $settings);
    }
    
    /**
     * اضافه کردن ادمین جدید
     */
    public static function addAdmin($userId, $firstName = '', $username = '') {
        $settings = self::getSettings();
        
        if (!isset($settings['admins_list'])) {
            $settings['admins_list'] = [];
        }
        
        // بررسی تکراری نبودن
        if (in_array($userId, $settings['admins_list'])) {
            return false; // قبلاً ادمین است
        }
        
        $settings['admins_list'][] = $userId;
        self::write(DATA_PATH . 'settings.json', $settings);
        
        return true;
    }
    
    /**
     * حذف ادمین
     */
    public static function removeAdmin($userId) {
        $userId = intval($userId);
        
        if ($userId <= 0) {
            return false;
        }
        
        $settings = self::getSettings();
        
        if (!isset($settings['admins_list']) || !is_array($settings['admins_list'])) {
            return false;
        }
        
        $key = array_search($userId, $settings['admins_list'], true);
        if ($key === false) {
            return false; // ادمین نیست
        }
        
        unset($settings['admins_list'][$key]);
        $settings['admins_list'] = array_values($settings['admins_list']); // reindex
        
        self::write(DATA_PATH . 'settings.json', $settings);
        return true;
    }
    
    /**
     * لیست همه ادمین‌ها
     */
    public static function getAdminsList() {
        $settings = self::getSettings();
        return $settings['admins_list'] ?? [];
    }
    
    /**
     * بررسی اینکه کاربر ادمین است
     */
    public static function isAdmin($userId) {
        // ادمین اصلی
        if ($userId == ADMIN_ID) {
            return true;
        }
        
        // ادمین‌های اضافه شده
        $adminsList = self::getAdminsList();
        return in_array($userId, $adminsList);
    }
    
    /**
     * اضافه کردن کانال به لیست جوین اجباری
     */
    public static function addForceJoinChannel($channelUsername, $channelId, $channelTitle = '') {
        $settings = self::getSettings();
        
        if (!isset($settings['force_join_channels'])) {
            $settings['force_join_channels'] = [];
        }
        
        // بررسی تکراری نبودن
        foreach ($settings['force_join_channels'] as $channel) {
            if ($channel['id'] == $channelId) {
                return false; // قبلاً اضافه شده
            }
        }
        
        $settings['force_join_channels'][] = [
            'username' => $channelUsername,
            'id' => $channelId,
            'title' => $channelTitle,
            'added_at' => date('Y-m-d H:i:s')
        ];
        
        self::write(DATA_PATH . 'settings.json', $settings);
        return true;
    }
    
    /**
     * حذف کانال از لیست جوین اجباری
     */
    public static function removeForceJoinChannel($channelId) {
        $settings = self::getSettings();
        
        if (!isset($settings['force_join_channels'])) {
            return false;
        }
        
        $newChannels = [];
        $found = false;
        
        foreach ($settings['force_join_channels'] as $channel) {
            if ($channel['id'] != $channelId) {
                $newChannels[] = $channel;
            } else {
                $found = true;
            }
        }
        
        if ($found) {
            $settings['force_join_channels'] = $newChannels;
            self::write(DATA_PATH . 'settings.json', $settings);
            return true;
        }
        
        return false;
    }
    
    /**
     * دریافت لیست کانال‌های جوین اجباری
     */
    public static function getForceJoinChannels() {
        $settings = self::getSettings();
        return isset($settings['force_join_channels']) ? $settings['force_join_channels'] : [];
    }
    
    /**
     * بروزرسانی تنظیمات
     */
    public static function updateSetting($key, $value) {
        $settings = self::getSettings();
        $settings[$key] = $value;
        return self::write(DATA_PATH . 'settings.json', $settings);
    }
    
    /**
     * دریافت یک تنظیم خاص
     */
    public static function getSetting($key, $default = null) {
        $settings = self::getSettings();
        return isset($settings[$key]) ? $settings[$key] : $default;
    }
    
    /**
     * بررسی رمز عبور فایل
     */
    public static function checkFilePassword($fileId, $password) {
        $file = self::getFile($fileId);
        
        if (!$file) {
            return false;
        }
        
        // اگر فایل رمز ندارد
        if (empty($file['password'])) {
            return true;
        }
        
        // مقایسه رمز
        return $file['password'] === $password;
    }
    
    /**
     * بررسی اینکه فایل رمز دارد یا نه
     */
    public static function fileHasPassword($fileId) {
        $file = self::getFile($fileId);
        return $file && !empty($file['password']);
    }
    
    /**
     * ذخیره رمز موقت برای چک کردن
     */
    public static function setPasswordAttempt($userId, $fileId, $password) {
        $attempts = self::read(UPLOAD_TEMP_DB);
        
        $attempts['password_' . $userId] = [
            'file_id' => $fileId,
            'password' => $password,
            'timestamp' => time()
        ];
        
        self::write(UPLOAD_TEMP_DB, $attempts);
    }
    
    /**
     * دریافت رمز موقت
     */
    public static function getPasswordAttempt($userId) {
        $attempts = self::read(UPLOAD_TEMP_DB);
        return isset($attempts['password_' . $userId]) ? $attempts['password_' . $userId] : null;
    }
    
    /**
     * پاک کردن رمز موقت
     */
    public static function clearPasswordAttempt($userId) {
        $attempts = self::read(UPLOAD_TEMP_DB);
        
        if (isset($attempts['password_' . $userId])) {
            unset($attempts['password_' . $userId]);
            self::write(UPLOAD_TEMP_DB, $attempts);
        }
    }
    
    /**
     * ثبت امتیاز فایل
     */
    public static function rateFile($fileId, $userId, $rating) {
        $files = self::read(FILES_DB);
        
        if (!isset($files[$fileId])) {
            return false;
        }
        
        // مقدار امتیاز باید بین 1 تا 5 باشد
        if ($rating < 1 || $rating > 5) {
            return false;
        }
        
        if (!isset($files[$fileId]['ratings'])) {
            $files[$fileId]['ratings'] = [];
        }
        
        // ثبت امتیاز کاربر
        $files[$fileId]['ratings'][$userId] = $rating;
        
        // محاسبه میانگین
        $totalRatings = count($files[$fileId]['ratings']);
        $sumRatings = array_sum($files[$fileId]['ratings']);
        $files[$fileId]['average_rating'] = round($sumRatings / $totalRatings, 1);
        $files[$fileId]['total_ratings'] = $totalRatings;
        
        self::write(FILES_DB, $files);
        return true;
    }
    
    /**
     * دریافت امتیاز کاربر برای فایل
     */
    public static function getUserRatingForFile($fileId, $userId) {
        $files = self::read(FILES_DB);
        
        if (!isset($files[$fileId]) || !isset($files[$fileId]['ratings'][$userId])) {
            return null;
        }
        
        return $files[$fileId]['ratings'][$userId];
    }
    
    /**
     * دریافت بهترین فایل‌ها
     */
    public static function getTopRatedFiles($limit = 10) {
        $files = self::getAllFiles();
        
        // فیلتر فایل‌های فعال با امتیاز
        $ratedFiles = array_filter($files, function($file) {
            return $file['is_active'] && isset($file['total_ratings']) && $file['total_ratings'] > 0;
        });
        
        // مرتب‌سازی بر اساس میانگین امتیاز
        usort($ratedFiles, function($a, $b) {
            if ($a['average_rating'] == $b['average_rating']) {
                // اگر میانگین برابر بود، بر اساس تعداد رأی‌دهندگان
                return $b['total_ratings'] - $a['total_ratings'];
            }
            return $b['average_rating'] <=> $a['average_rating'];
        });
        
        return array_slice($ratedFiles, 0, $limit);
    }
    
    /**
     * آمار کلی
     */
    public static function getStats() {
        $files = self::getAllFiles();
        $users = self::read(USERS_DB);
        
        $totalDownloads = 0;
        $totalSize = 0;
        
        foreach ($files as $file) {
            $totalDownloads += $file['download_count'];
            $totalSize += $file['file_size'];
        }
        
        return [
            'total_files' => count($files),
            'total_users' => count($users),
            'total_downloads' => $totalDownloads,
            'total_size' => $totalSize
        ];
    }
    
    /**
     * ثبت فعالیت جدید
     */
    public static function logActivity($type, $title, $details = []) {
        $activities = self::read(ACTIVITY_DB);
        
        if (!is_array($activities)) {
            $activities = [];
        }
        
        $activity = [
            'id' => uniqid(),
            'type' => $type,
            'title' => $title,
            'details' => $details,
            'time' => time(),
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        // اضافه کردن در ابتدای آرایه
        array_unshift($activities, $activity);
        
        // نگه داشتن فقط 100 فعالیت آخر
        $activities = array_slice($activities, 0, 100);
        
        self::write(ACTIVITY_DB, $activities);
        
        return true;
    }
    
    /**
     * دریافت فعالیت‌های اخیر
     */
    public static function getRecentActivities($limit = 10) {
        $activities = self::read(ACTIVITY_DB);
        
        if (!is_array($activities)) {
            return [];
        }
        
        return array_slice($activities, 0, $limit);
    }
}
