<?php
/**
 * سیستم آپلود پیشرفته
 * Advanced Upload System Handler
 */

/**
 * نمایش منوی تنظیمات فایل
 */
function showFileSettings($chatId, $userId) {
    $state = Database::getUploadState($userId);
    
    if (!$state || !isset($state['data'])) {
        Telegram::sendMessage($chatId, "❌ خطا! لطفاً فایل را دوباره ارسال کنید.");
        return;
    }
    
    $fileData = $state['data'];
    
    $message = "⚙️ " . HTMLParser::format("تنظیمات فایل", true) . "\n\n";
    $message .= "📁 " . HTMLParser::format($fileData['file_name'], false, false, true) . "\n";
    $message .= "💾 " . HTMLParser::formatFileSize($fileData['file_size']) . "\n\n";
    
    $message .= HTMLParser::blockquote("لطفاً تنظیمات فایل را مشخص کنید");
    
    $keyboard = Telegram::makeInlineKeyboard([
        [
            ['text' => '🏷 افزودن تایتل', 'callback_data' => 'set_title']
        ],
        [
            ['text' => '💬 افزودن کپشن', 'callback_data' => 'set_caption']
        ],
        [
            ['text' => '� تنظیم رمز عبور', 'callback_data' => 'set_password']
        ],
        [
            ['text' => '�🔢 محدودیت دانلود', 'callback_data' => 'set_limit']
        ],
        [            ['text' => '⏰ تاریخ انقضا', 'callback_data' => 'set_expire']
        ],
        [            ['text' => '🔗 جوین اجباری: ✅ فعال', 'callback_data' => 'toggle_force_join']
        ],
        [
            ['text' => '👁 پیشنمایش', 'callback_data' => 'preview_file'],
            ['text' => '✅ تایید و آپلود', 'callback_data' => 'confirm_upload']
        ],
        [
            ['text' => '❌ لغو', 'callback_data' => 'cancel_upload']
        ]
    ]);
    
    Telegram::sendMessage($chatId, $message, $keyboard);
}

/**
 * مدیریت آپلود از تلگرام
 */
function handleTelegramUpload($chatId, $userId) {
    Database::setUploadState($userId, 'awaiting_telegram_file', []);
    
    $message = "📎 " . HTMLParser::format("آپلود از تلگرام", true) . "\n\n";
    $message .= HTMLParser::blockquote("فایل مورد نظر خود را ارسال کنید");
    $message .= "\n\n💡 انواع فایل پشتیبانی شده:\n";
    $message .= "• عکس، ویدیو، صوت\n";
    $message .= "• اسناد (PDF, ZIP, ...)\n";
    $message .= "• حداکثر حجم: " . HTMLParser::formatFileSize(MAX_FILE_SIZE);
    $message .= "\n\nبرای لغو: /cancel";
    
    Telegram::sendMessage($chatId, $message, Telegram::makeKeyboard([
        [
            ['text' => '🔙 بازگشت']
        ]
    ]));
}

/**
 * راهنمای آپلود از لینک
 */
function handleLinkUploadGuide($chatId, $userId) {
    Database::setUploadState($userId, 'awaiting_link', []);
    
    $message = "🔗 " . HTMLParser::format("آپلود از لینک", true) . "\n\n";
    $message .= HTMLParser::blockquote("لینک مستقیم دانلود فایل را ارسال کنید");
    $message .= "\n\n📝 مثال:\n";
    $message .= HTMLParser::format("https://example.com/file.zip", false, false, true);
    $message .= "\n\n💡 نکات:\n";
    $message .= "• لینک باید مستقیم باشد\n";
    $message .= "• حداکثر حجم: " . HTMLParser::formatFileSize(MAX_FILE_SIZE);
    $message .= "\n\nبرای لغو: /cancel";
    
    Telegram::sendMessage($chatId, $message, Telegram::makeKeyboard([
        [
            ['text' => '🔙 بازگشت']
        ]
    ]));
}

/**
 * هندلر آپلود از لینک
 */
function handleUploadFromUrl($chatId, $userId, $url) {
    if (!Telegram::isAdmin($userId)) {
        return;
    }
    
    // بررسی اعتبار URL
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        Telegram::sendMessage($chatId, "❌ لینک نامعتبر است!");
        return;
    }
    
    $progressMsg = "📥 " . HTMLParser::format("دانلود فایل...", true) . "\n\n";
    $progressMsg .= HTMLParser::blockquote("لطفاً صبر کنید");
    
    $msgResult = Telegram::sendMessage($chatId, $progressMsg);
    $messageId = $msgResult['result']['message_id'] ?? null;
    
    // دریافت اطلاعات فایل
    $fileInfo = FileUploader::getFileInfoFromUrl($url);
    
    if (!$fileInfo) {
        Telegram::sendMessage($chatId, "❌ خطا در دریافت اطلاعات فایل!");
        return;
    }
    
    // بررسی حجم
    if ($fileInfo['size'] > MAX_FILE_SIZE) {
        $errorMsg = "❌ حجم فایل بیش از حد مجاز است!\n\n";
        $errorMsg .= "حجم فایل: " . HTMLParser::formatFileSize($fileInfo['size']) . "\n";
        $errorMsg .= "حداکثر: " . HTMLParser::formatFileSize(MAX_FILE_SIZE);
        Telegram::sendMessage($chatId, $errorMsg);
        return;
    }
    
    // دانلود فایل
    $tempFile = UPLOADS_PATH . 'temp_' . time() . '_' . $fileInfo['name'];
    
    if (!FileUploader::downloadFromUrl($url, $tempFile)) {
        Telegram::sendMessage($chatId, "❌ خطا در دانلود فایل!");
        return;
    }
    
    // آپلود به تلگرام
    $updateMsg = "📤 " . HTMLParser::format("آپلود به تلگرام...", true) . "\n\n";
    $updateMsg .= HTMLParser::blockquote("در حال پردازش");
    
    if ($messageId) {
        Telegram::request('editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $updateMsg,
            'parse_mode' => 'HTML'
        ]);
    }
    
    $fileType = FileUploader::detectFileType($fileInfo['name']);
    $uploadResult = FileUploader::uploadToTelegram($chatId, $tempFile, $fileType);
    
    @unlink($tempFile);
    
    if (isset($uploadResult['ok']) && $uploadResult['ok']) {
        // استخراج file_id
        $result = $uploadResult['result'];
        $fileId = null;
        
        if (isset($result['document'])) {
            $fileId = $result['document']['file_id'];
        } elseif (isset($result['photo'])) {
            $photos = $result['photo'];
            $photo = end($photos);
            $fileId = $photo['file_id'];
        } elseif (isset($result['video'])) {
            $fileId = $result['video']['file_id'];
        } elseif (isset($result['audio'])) {
            $fileId = $result['audio']['file_id'];
        }
        
        if ($fileId) {
            // ذخیره اطلاعات موقت
            Database::setUploadState($userId, 'awaiting_settings', [
                'file_id' => $fileId,
                'file_name' => $fileInfo['name'],
                'file_type' => $fileType,
                'file_size' => filesize($tempFile) ?: $fileInfo['size']
            ]);
            
            showFileSettings($chatId, $userId);
        }
    } else {
        Telegram::sendMessage($chatId, "❌ خطا در آپلود فایل به تلگرام!");
    }
}

/**
 * پردازش callback های تنظیمات فایل
 */
function handleFileSettingsCallback($chatId, $userId, $data, $messageId) {
    $state = Database::getUploadState($userId);
    
    if (!$state) {
        Telegram::request('answerCallbackQuery', [
            'callback_query_id' => $data,
            'text' => 'خطا! لطفاً دوباره فایل را ارسال کنید.'
        ]);
        return;
    }
    
    $fileData = $state['data'];
    
    switch ($data) {
        case 'set_title':
            Database::setUploadState($userId, 'awaiting_title', $fileData);
            
            $msg = "🏷 " . HTMLParser::format("افزودن تایتل", true) . "\n\n";
            $msg .= HTMLParser::blockquote("تایتل (عنوان) دلخواه خود را ارسال کنید");
            $msg .= "\n\n💡 تایتل برای نمایش بهتر در لیست فایل‌ها استفاده می‌شود";
            $msg .= "\nبرای لغو: /cancel";
            
            Telegram::sendMessage($chatId, $msg);
            break;
            
        case 'set_caption':
            Database::setUploadState($userId, 'awaiting_caption', $fileData);
            
            $msg = "💬 " . HTMLParser::format("افزودن کپشن", true) . "\n\n";
            $msg .= HTMLParser::blockquote("کپشن دلخواه خود را ارسال کنید");
            $msg .= "\n\nبرای لغو: /cancel";
            
            Telegram::sendMessage($chatId, $msg);
            break;
            
        case 'set_password':
            Database::setUploadState($userId, 'awaiting_password', $fileData);
            
            $msg = "🔐 " . HTMLParser::format("تنظیم رمز عبور", true) . "\n\n";
            $msg .= HTMLParser::blockquote("رمز عبور دلخواه خود را ارسال کنید");
            $msg .= "\n\n🔒 کاربران باید این رمز را وارد کنند تا فایل را دانلود کنند";
            $msg .= "\nبرای حذف رمز: " . HTMLParser::format("0", false, false, true);
            $msg .= "\nبرای لغو: /cancel";
            
            Telegram::sendMessage($chatId, $msg);
            break;
            
        case 'set_limit':
            Database::setUploadState($userId, 'awaiting_limit', $fileData);
            
            $msg = "🔢 " . HTMLParser::format("محدودیت دانلود", true) . "\n\n";
            $msg .= HTMLParser::blockquote("تعداد دفعات مجاز دانلود را وارد کنید");
            $msg .= "\n\nبرای نامحدود: " . HTMLParser::format("0", false, false, true);
            $msg .= "\nبرای لغو: /cancel";
            
            Telegram::sendMessage($chatId, $msg);
            break;
            
        case 'set_expire':
            Database::setUploadState($userId, 'awaiting_expire', $fileData);
            
            $msg = "⏰ " . HTMLParser::format("تاریخ انقضا", true) . "\n\n";
            $msg .= HTMLParser::blockquote("تعداد روز تا انقضا را وارد کنید");
            $msg .= "\n\nمثال: " . HTMLParser::format("7", false, false, true) . " (برای 7 روز)";
            $msg .= "\nبرای بدون انقضا: " . HTMLParser::format("0", false, false, true);
            $msg .= "\nبرای لغو: /cancel";
            
            Telegram::sendMessage($chatId, $msg);
            break;
            
        case 'toggle_force_join':
            $fileData['force_join_enabled'] = !($fileData['force_join_enabled'] ?? true);
            Database::setUploadState($userId, 'awaiting_settings', $fileData);
            
            updateFileSettingsMessage($chatId, $messageId, $fileData);
            break;
            
        case 'preview_file':
            showFilePreview($chatId, $fileData);
            break;
            
        case 'confirm_upload':
            confirmAndUpload($chatId, $userId, $fileData);
            break;
            
        case 'cancel_upload':
            Database::clearUploadState($userId);
            Telegram::sendMessage($chatId, "❌ آپلود لغو شد.");
            break;
    }
}

/**
 * بروزرسانی پیام تنظیمات
 */
function updateFileSettingsMessage($chatId, $messageId, $fileData) {
    $forceJoinText = ($fileData['force_join_enabled'] ?? true) ? '✅ فعال' : '❌ غیرفعال';
    
    $message = "⚙️ " . HTMLParser::format("تنظیمات فایل", true) . "\n\n";
    $message .= "📁 " . HTMLParser::format($fileData['file_name'], false, false, true) . "\n";
    $message .= "💾 " . HTMLParser::formatFileSize($fileData['file_size']) . "\n";
    
    if (!empty($fileData['title'])) {
        $message .= "🏷 تایتل: ✅\n";
    }
    if (!empty($fileData['caption'])) {
        $message .= "💬 کپشن: ✅\n";
    }
    if (!empty($fileData['password'])) {
        $message .= "🔐 رمز عبور: ✅\n";
    }
    if (isset($fileData['download_limit']) && $fileData['download_limit'] > 0) {
        $message .= "🔢 محدودیت: " . $fileData['download_limit'] . "\n";
    }
    if (isset($fileData['expire_days']) && $fileData['expire_days'] > 0) {
        $message .= "⏰ انقضا: " . $fileData['expire_days'] . " روز\n";
    }
    
    $message .= "\n" . HTMLParser::blockquote("لطفاً تنظیمات فایل را مشخص کنید");
    
    $keyboard = Telegram::makeInlineKeyboard([
        [
            ['text' => '💬 افزودن کپشن', 'callback_data' => 'set_caption']
        ],
        [
            ['text' => '🔢 محدودیت دانلود', 'callback_data' => 'set_limit']
        ],
        [            ['text' => '⏰ تاریخ انقضا', 'callback_data' => 'set_expire']
        ],
        [            ['text' => '🔗 جوین اجباری: ' . $forceJoinText, 'callback_data' => 'toggle_force_join']
        ],
        [
            ['text' => '👁 پیشنمایش', 'callback_data' => 'preview_file'],
            ['text' => '✅ تایید و آپلود', 'callback_data' => 'confirm_upload']
        ],
        [
            ['text' => '❌ لغو', 'callback_data' => 'cancel_upload']
        ]
    ]);
    
    Telegram::request('editMessageText', [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode($keyboard)
    ]);
}

/**
 * نمایش پیشنمایش فایل
 */
function showFilePreview($chatId, $fileData) {
    $preview = FileUploader::generatePreview($fileData);
    Telegram::sendMessage($chatId, $preview);
}

/**
 * تایید و آپلود نهایی فایل
 */
function confirmAndUpload($chatId, $userId, $fileData) {
    $expireDate = null;
    
    if (isset($fileData['expire_days']) && $fileData['expire_days'] > 0) {
        $expireDate = date('Y-m-d H:i:s', strtotime('+' . $fileData['expire_days'] . ' days'));
    }
    
    $options = [
        'title' => $fileData['title'] ?? '',
        'caption' => $fileData['caption'] ?? '',
        'download_limit' => $fileData['download_limit'] ?? 0,
        'force_join_enabled' => $fileData['force_join_enabled'] ?? true,
        'expire_date' => $expireDate,
        'password' => $fileData['password'] ?? ''
    ];
    
    $uniqueId = Database::addFile(
        $fileData['file_id'],
        $fileData['file_name'],
        $fileData['file_type'],
        $fileData['file_size'],
        $userId,
        $options
    );
    
    Database::clearUploadState($userId);
    
    // ارسال اطلاع‌رسانی آپلود جدید
    sendNewFileNotification($fileData, $uniqueId);
    
    // ساخت لینک
    $botUsername = BOT_USERNAME;
    $link = "https://t.me/{$botUsername}?start={$uniqueId}";
    
    $successMsg = "✅ " . HTMLParser::format("فایل با موفقیت آپلود شد!", true) . "\n\n";
    $successMsg .= "📁 نام: " . HTMLParser::format($fileData['file_name'], false, false, true) . "\n";
    $successMsg .= "💾 حجم: " . HTMLParser::format(HTMLParser::formatFileSize($fileData['file_size'])) . "\n";
    $successMsg .= "🔑 ID: " . HTMLParser::format($uniqueId, false, false, true) . "\n\n";
    
    $successMsg .= HTMLParser::blockquote("لینک دانلود") . "\n";
    $successMsg .= HTMLParser::format($link, false, false, true) . "\n\n";
    
    if ($options['download_limit'] > 0) {
        $successMsg .= "🔢 محدودیت: " . $options['download_limit'] . " دانلود\n";
    }
    
    $forceJoinStatus = $options['force_join_enabled'] ? 'فعال' : 'غیرفعال';
    $successMsg .= "🔗 جوین اجباری: " . $forceJoinStatus;
    
    Telegram::sendMessage($chatId, $successMsg, Telegram::getAdminKeyboard());
}
