<?php
/**
 * Admin Panel API
 */

// Disable all PHP errors
@ini_set('display_errors', '0');
@ini_set('display_startup_errors', '0');
@ini_set('log_errors', '0');
@error_reporting(0);

// Start output buffering
ob_start();

// Set headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ob_end_clean();
    exit(0);
}

// Response helper
function sendResponse($success, $data = [], $message = '', $token = null) {
    while (ob_get_level()) {
        ob_end_clean();
    }
    
    ob_start();
    $response = [
        'success' => $success,
        'message' => $message,
        'data' => $data
    ];
    
    if ($token !== null) {
        $response['token'] = $token;
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    ob_end_flush();
    exit;
}

// Load config
$configPath = __DIR__ . '/config.php';
if (!file_exists($configPath)) {
    sendResponse(false, [], 'Configuration file not found');
}

require_once $configPath;
require_once __DIR__ . '/includes/database.php';
require_once __DIR__ . '/includes/Telegram.php';

// Get request data
$input = @file_get_contents('php://input');
$request = @json_decode($input, true);

// If JSON decode fails, try to parse as form data
if (!$request) {
    parse_str($input, $request);
    if (empty($request)) {
        $request = $_POST;
    }
}

if (!$request || !isset($request['action'])) {
    sendResponse(false, [], 'Invalid request');
}

$action = $request['action'];

// Login actions (don't need token)
if ($action === 'login') {
    handleLogin($request);
} elseif ($action === 'loginWithTelegram') {
    handleLoginWithTelegram($request);
}

// Verify token for all other actions
$token = $request['token'] ?? '';
if (!verifyAdminToken($token)) {
    sendResponse(false, [], 'Unauthorized');
}

// Handle actions
switch ($action) {
    case 'verifyToken':
        handleVerifyToken($token);
        break;
    
    case 'getStats':
        handleGetStats();
        break;
    
    case 'getUsers':
        handleGetUsers();
        break;
    
    case 'getUserDetails':
        handleGetUserDetails($request);
        break;
    
    case 'blockUser':
        handleBlockUser($request);
        break;
    
    case 'unblockUser':
        handleUnblockUser($request);
        break;
    
    case 'sendWarning':
        handleSendWarning($request);
        break;
    
    case 'deleteWarning':
        handleDeleteWarning($request);
        break;
    
    case 'getFiles':
        handleGetFiles();
        break;
    
    case 'updateFile':
        handleUpdateFile($request);
        break;
    
    case 'deleteFile':
        handleDeleteFile($request);
        break;
    
    case 'sendBroadcast':
        handleSendBroadcast($request);
        break;
    
    case 'getAppeals':
        handleGetAppeals();
        break;
    
    case 'acceptAppeal':
        handleAcceptAppeal($request);
        break;
    
    case 'rejectAppeal':
        handleRejectAppeal($request);
        break;
    
    case 'checkUserType':
        handleCheckUserType($request);
        break;
    
    case 'getAdmins':
        handleGetAdmins();
        break;
    
    case 'addAdmin':
        handleAddAdmin($request);
        break;
    
    case 'removeAdmin':
        handleRemoveAdmin($request);
        break;
    
    case 'getWebhookInfo':
        handleGetWebhookInfo();
        break;
    
    case 'setWebhook':
        handleSetWebhook($request);
        break;
    
    case 'deleteWebhook':
        handleDeleteWebhook();
        break;
    
    case 'getForceJoinChannels':
        handleGetForceJoinChannels();
        break;
    
    case 'checkChannelInfo':
        handleCheckChannelInfo($request);
        break;
    
    case 'addForceJoinChannel':
        handleAddForceJoinChannel($request);
        break;
    
    case 'updateForceJoinChannel':
        handleUpdateForceJoinChannel($request);
        break;
    
    case 'deleteForceJoinChannel':
        handleDeleteForceJoinChannel($request);
        break;
    
    default:
        sendResponse(false, [], 'Unknown action');
}

// Functions

function handleLogin($request) {
    $username = $request['username'] ?? '';
    $password = $request['password'] ?? '';
    
    // Simple authentication - you should change this to your own system
    // You can check against ADMIN_ID from config or a database
    if ($username === 'admin' && $password === 'admin123') {
        $token = bin2hex(random_bytes(32));
        $adminId = defined('ADMIN_ID') ? ADMIN_ID : 123456789;
        
        // Save token (in production, use database)
        $tokensFile = __DIR__ . '/data/admin_tokens.json';
        $tokens = [];
        if (file_exists($tokensFile)) {
            $tokens = json_decode(file_get_contents($tokensFile), true) ?: [];
        }
        
        $tokens[$token] = [
            'admin_id' => $adminId,
            'expires' => time() + (7 * 24 * 3600) // 7 days
        ];
        
        @mkdir(__DIR__ . '/data', 0755, true);
        file_put_contents($tokensFile, json_encode($tokens));
        
        // Get admin info
        $db = new Database();
        $admin = $db->getUser($adminId);
        
        $response = [
            'success' => true,
            'token' => $token,
            'data' => [
                'first_name' => $admin['first_name'] ?? 'ادمین',
                'is_admin' => true
            ]
        ];
        
        while (ob_get_level()) {
            ob_end_clean();
        }
        ob_start();
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        ob_end_flush();
        exit;
    } else {
        sendResponse(false, [], 'Invalid credentials');
    }
}

function handleLoginWithTelegram($request) {
    $telegramData = $request['telegram_data'] ?? [];
    $userId = intval($telegramData['id'] ?? 0);
    
    if (!$userId) {
        sendResponse(false, [], 'Invalid Telegram data');
    }
    
    // Check if user is admin
    $isAdmin = Database::isAdmin($userId);
    
    // For test user, always allow
    if ($userId === 987654321) {
        $isAdmin = true;
        
        // Save test admin to database if not exists
        $user = Database::getUser($userId);
        if (!$user) {
            Database::addUser(
                $userId,
                $telegramData['username'] ?? 'test_admin',
                $telegramData['first_name'] ?? 'ادمین',
                $telegramData['last_name'] ?? 'تست'
            );
            
            // Add as admin
            Database::addAdmin($userId, $telegramData['first_name'] ?? 'ادمین', $telegramData['username'] ?? 'test_admin');
        }
    }
    
    if (!$isAdmin) {
        sendResponse(false, [], 'شما دسترسی ادمین ندارید');
    }
    
    // Generate token
    $token = bin2hex(random_bytes(32));
    
    $tokensFile = __DIR__ . '/data/admin_tokens.json';
    $tokens = [];
    if (file_exists($tokensFile)) {
        $tokens = json_decode(file_get_contents($tokensFile), true) ?: [];
    }
    
    $tokens[$token] = [
        'admin_id' => $userId,
        'expires' => time() + (7 * 24 * 3600) // 7 days
    ];
    
    @mkdir(__DIR__ . '/data', 0755, true);
    file_put_contents($tokensFile, json_encode($tokens));
    
    // Return success with token
    sendResponse(true, [
        'first_name' => $telegramData['first_name'] ?? 'ادمین',
        'last_name' => $telegramData['last_name'] ?? '',
        'username' => $telegramData['username'] ?? '',
        'is_admin' => true
    ], '', $token);
}

function verifyAdminToken($token) {
    if (!$token) return false;
    
    $tokensFile = __DIR__ . '/data/admin_tokens.json';
    if (!file_exists($tokensFile)) return false;
    
    $tokens = json_decode(file_get_contents($tokensFile), true) ?: [];
    
    if (!isset($tokens[$token])) return false;
    
    if ($tokens[$token]['expires'] < time()) {
        unset($tokens[$token]);
        file_put_contents($tokensFile, json_encode($tokens));
        return false;
    }
    
    return true;
}

function handleVerifyToken($token) {
    $tokensFile = __DIR__ . '/data/admin_tokens.json';
    $tokens = json_decode(file_get_contents($tokensFile), true) ?: [];
    
    $adminId = $tokens[$token]['admin_id'];
    $db = new Database();
    $admin = $db->getUser($adminId);
    
    sendResponse(true, [
        'first_name' => $admin['first_name'] ?? 'ادمین',
        'is_admin' => true
    ]);
}

function handleGetStats() {
    $db = new Database();
    $users = $db->getAllUsers();
    $files = Database::getAllFiles();
    
    $totalUsers = count($users);
    $activeUsers = 0;
    $blockedUsers = 0;
    $totalWarnings = 0;
    $totalDownloads = 0;
    
    foreach ($users as $user) {
        if ($user['is_blocked']) {
            $blockedUsers++;
        } else {
            $activeUsers++;
        }
        $totalWarnings += $user['warning_count'] ?? 0;
        $totalDownloads += $user['download_count'] ?? 0;
    }
    
    // Count real files
    $totalFiles = count($files);
    
    // Get recent activities from database
    $recentActivity = Database::getRecentActivities(10);
    
    sendResponse(true, [
        'total_users' => $totalUsers,
        'active_users' => $activeUsers,
        'blocked_users' => $blockedUsers,
        'total_files' => $totalFiles,
        'total_downloads' => $totalDownloads,
        'total_warnings' => $totalWarnings,
        'recent_activity' => $recentActivity
    ]);
}

function handleGetUsers() {
    $db = new Database();
    $users = $db->getAllUsers();
    
    sendResponse(true, $users);
}

function handleGetUserDetails($request) {
    $userId = intval($request['user_id']);
    $db = new Database();
    $user = $db->getUser($userId);
    
    if (!$user) {
        sendResponse(false, [], 'User not found');
    }
    
    sendResponse(true, $user);
}

function handleBlockUser($request) {
    $userId = intval($request['user_id']);
    $db = new Database();
    $user = $db->getUser($userId);
    
    $db->blockUser($userId);
    
    // Log activity
    Database::logActivity('user_block', 'کاربر مسدود شد', [
        'user_id' => $userId,
        'name' => $user['first_name'] ?? 'کاربر'
    ]);
    
    sendResponse(true, [], 'User blocked successfully');
}

function handleUnblockUser($request) {
    $userId = intval($request['user_id']);
    $db = new Database();
    $user = $db->getUser($userId);
    
    $db->unblockUser($userId);
    
    // Log activity
    Database::logActivity('user_unblock', 'کاربر رفع مسدودیت شد', [
        'user_id' => $userId,
        'name' => $user['first_name'] ?? 'کاربر'
    ]);
    
    sendResponse(true, [], 'User unblocked successfully');
}

function handleSendWarning($request) {
    $userId = intval($request['user_id']);
    $reason = $request['reason'] ?? 'تخلف از قوانین';
    
    $db = new Database();
    $db->addWarning($userId, ADMIN_ID, $reason);
    
    // Send notification to user via Telegram
    try {
        $user = $db->getUser($userId);
        
        // Log activity
        Database::logActivity('warning', 'اخطار جدید صادر شد', [
            'user_id' => $userId,
            'name' => $user['first_name'] ?? 'کاربر',
            'reason' => $reason
        ]);
        $warningCount = $user['warning_count'];
        
        $message = "⚠️ اخطار جدید\n\n";
        $message .= "دلیل: {$reason}\n\n";
        $message .= "تعداد اخطارهای شما: {$warningCount}\n";
        
        if ($warningCount >= 3) {
            $message .= "\n⛔️ حساب شما مسدود شد.";
            $db->blockUser($userId);
        }
        
        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
        $data = [
            'chat_id' => $userId,
            'text' => $message
        ];
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_exec($ch);
        curl_close($ch);
    } catch (Exception $e) {
        // Ignore errors
    }
    
    sendResponse(true, [], 'Warning sent successfully');
}

function handleDeleteWarning($request) {
    $userId = intval($request['user_id']);
    $warningIndex = intval($request['warning_index']);
    
    $db = new Database();
    $user = $db->getUser($userId);
    
    if (!$user) {
        sendResponse(false, [], 'User not found');
        return;
    }
    
    // حذف اخطار از آرایه
    if (isset($user['warnings'][$warningIndex])) {
        array_splice($user['warnings'], $warningIndex, 1);
        $user['warning_count'] = count($user['warnings']);
        
        // ذخیره تغییرات
        $users = $db->read(USERS_DB);
        $users[$userId] = $user;
        $db->write(USERS_DB, $users);
        
        sendResponse(true, [], 'Warning deleted successfully');
    } else {
        sendResponse(false, [], 'Warning not found');
    }
}

function handleGetFiles() {
    $db = new Database();
    $files = $db->getAllFiles();
    
    // Format files for response
    $formattedFiles = [];
    foreach ($files as $file) {
        $formattedFiles[] = [
            'id' => $file['file_id'] ?? '',
            'file_id' => $file['file_id'] ?? '',
            'file_name' => $file['file_name'] ?? '',
            'title' => $file['title'] ?? '',
            'caption' => $file['caption'] ?? '',
            'file_type' => $file['file_type'] ?? 'unknown',
            'file_size' => $file['file_size'] ?? 0,
            'upload_date' => $file['upload_date'] ?? '',
            'uploader_id' => $file['uploader_id'] ?? '',
            'download_count' => $file['download_count'] ?? 0,
            'download_limit' => $file['download_limit'] ?? 0,
            'expire_date' => $file['expire_date'] ?? '',
            'has_password' => $file['has_password'] ?? false,
            'password' => $file['password'] ?? '',
            'is_active' => $file['is_active'] ?? true
        ];
    }
    
    sendResponse(true, $formattedFiles);
}

function handleUpdateFile($request) {
    $fileId = $request['file_id'] ?? '';
    
    if (empty($fileId)) {
        sendResponse(false, [], 'File ID is required');
    }
    
    $updates = [];
    
    if (isset($request['file_name'])) $updates['file_name'] = $request['file_name'];
    if (isset($request['title'])) $updates['title'] = $request['title'];
    if (isset($request['caption'])) $updates['caption'] = $request['caption'];
    if (isset($request['password'])) {
        $updates['password'] = $request['password'];
        $updates['has_password'] = !empty($request['password']);
    }
    if (isset($request['is_active'])) $updates['is_active'] = (bool)$request['is_active'];
    if (isset($request['expire_date'])) $updates['expire_date'] = $request['expire_date'];
    if (isset($request['download_limit'])) $updates['download_limit'] = intval($request['download_limit']);
    
    if (empty($updates)) {
        sendResponse(false, [], 'No fields to update');
    }
    
    $db = new Database();
    $result = $db->updateFile($fileId, $updates);
    
    if ($result) {
        sendResponse(true, [], 'File updated successfully');
    } else {
        sendResponse(false, [], 'File not found or update failed');
    }
}

function handleDeleteFile($request) {
    $fileId = $request['file_id'] ?? '';
    
    if (empty($fileId)) {
        sendResponse(false, [], 'File ID is required');
    }
    
    $db = new Database();
    
    // Get file info to delete physical file
    $file = $db->getFile($fileId);
    if ($file && !empty($file['file_path'])) {
        $filePath = __DIR__ . '/' . $file['file_path'];
        if (file_exists($filePath)) {
            @unlink($filePath);
        }
    }
    
    $result = $db->deleteFile($fileId);
    
    if ($result) {
        sendResponse(true, [], 'File deleted successfully');
    } else {
        sendResponse(false, [], 'File not found or delete failed');
    }
}

function handleSendBroadcast($request) {
    $message = $request['message'] ?? '';
    $sendToAll = $request['send_to_all'] ?? true;
    $sendToActive = $request['send_to_active'] ?? false;
    
    if (!$message) {
        sendResponse(false, [], 'Message is required');
    }
    
    $db = new Database();
    $users = $db->getAllUsers();
    
    $sentCount = 0;
    foreach ($users as $user) {
        if ($sendToActive && $user['is_blocked']) continue;
        
        try {
            $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
            $data = [
                'chat_id' => $user['user_id'],
                'text' => $message
            ];
            
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
            
            if ($result) $sentCount++;
            
            usleep(100000); // 100ms delay between messages
        } catch (Exception $e) {
            continue;
        }
    }
    
    sendResponse(true, ['sent_count' => $sentCount], 'Broadcast sent successfully');
}

function handleGetAppeals() {
    $appealsFile = __DIR__ . '/data/appeals.json';
    $appeals = [];
    
    if (file_exists($appealsFile)) {
        $appeals = json_decode(file_get_contents($appealsFile), true) ?: [];
    }
    
    // Filter pending appeals
    $appeals = array_filter($appeals, function($appeal) {
        return ($appeal['status'] ?? 'pending') === 'pending';
    });
    
    sendResponse(true, array_values($appeals));
}

function handleAcceptAppeal($request) {
    $appealId = $request['appeal_id'] ?? '';
    $userId = intval($request['user_id']);
    
    // Unblock user
    $db = new Database();
    $db->unblockUser($userId);
    
    // Update appeal status
    $appealsFile = __DIR__ . '/data/appeals.json';
    if (file_exists($appealsFile)) {
        $appeals = json_decode(file_get_contents($appealsFile), true) ?: [];
        
        foreach ($appeals as &$appeal) {
            if ($appeal['id'] === $appealId) {
                $appeal['status'] = 'accepted';
                $appeal['reviewed_at'] = date('Y-m-d H:i:s');
                break;
            }
        }
        
        file_put_contents($appealsFile, json_encode($appeals, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    // Notify user
    try {
        $message = "✅ اعتراض شما پذیرفته شد\n\nمسدودیت حساب شما رفع گردید.";
        
        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
        $data = [
            'chat_id' => $userId,
            'text' => $message
        ];
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_exec($ch);
        curl_close($ch);
    } catch (Exception $e) {
        // Ignore
    }
    
    sendResponse(true, [], 'Appeal accepted');
}

function handleRejectAppeal($request) {
    $appealId = $request['appeal_id'] ?? '';
    $userId = intval($request['user_id'] ?? 0);
    $rejectReason = trim($request['reject_reason'] ?? 'اعتراض شما پذیرفته نشد.');
    
    $appealsFile = __DIR__ . '/data/appeals.json';
    if (file_exists($appealsFile)) {
        $appeals = json_decode(file_get_contents($appealsFile), true) ?: [];
        
        foreach ($appeals as &$appeal) {
            if ($appeal['id'] === $appealId) {
                $appeal['status'] = 'rejected';
                $appeal['reject_reason'] = $rejectReason;
                $appeal['reviewed_at'] = date('Y-m-d H:i:s');
                if (!$userId && isset($appeal['user_id'])) {
                    $userId = intval($appeal['user_id']);
                }
                break;
            }
        }
        
        file_put_contents($appealsFile, json_encode($appeals, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    // Notify user
    if ($userId > 0) {
        try {
            $message = "❌ اعتراض شما رد شد\n\n";
            $message .= "دلیل رد: " . $rejectReason . "\n\n";
            $message .= "شما می‌توانید مجدداً اعتراض کنید و مدارک بیشتری ارائه دهید.";
            
            $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
            $data = [
                'chat_id' => $userId,
                'text' => $message
            ];
            
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_exec($ch);
            curl_close($ch);
        } catch (Exception $e) {
            // Ignore
        }
    }
    
    sendResponse(true, [], 'Appeal rejected');
}

function handleGetAdmins() {
    $adminsList = Database::getAdminsList();
    $db = new Database();
    
    $admins = [];
    foreach ($adminsList as $userId) {
        $user = $db->getUser($userId);
        
        if ($user) {
            $admins[] = [
                'user_id' => $userId,
                'first_name' => $user['first_name'] ?? 'ادمین',
                'username' => $user['username'] ?? '',
                'is_main' => ($userId == ADMIN_ID)
            ];
        } else {
            // اگر کاربر در دیتابیس نباشد، فقط ID را نشان بده
            $admins[] = [
                'user_id' => $userId,
                'first_name' => 'ادمین',
                'username' => '',
                'is_main' => ($userId == ADMIN_ID)
            ];
        }
    }
    
    sendResponse(true, $admins);
}

function handleAddAdmin($request) {
    $userId = intval($request['user_id']);
    
    if ($userId <= 0) {
        sendResponse(false, [], 'شناسه کاربری نامعتبر است');
        return;
    }
    
    $db = new Database();
    
    // بررسی کنید که کاربر ربات را استارت کرده است
    $user = $db->getUser($userId);
    if (!$user) {
        sendResponse(false, [], 'این کاربر هنوز ربات را استارت نکرده است');
        return;
    }
    
    $result = $db->addAdmin($userId);
    
    if ($result) {
        // Log activity
        Database::logActivity('admin_add', 'ادمین جدید اضافه شد', [
            'user_id' => $userId,
            'name' => $user['first_name'] ?? 'کاربر'
        ]);
        
        sendResponse(true, [], 'Admin added successfully');
    } else {
        sendResponse(false, [], 'این کاربر قبلا ادمین است');
    }
}

function handleRemoveAdmin($request) {
    $userId = intval($request['user_id']);
    
    if ($userId <= 0) {
        sendResponse(false, [], 'Invalid user ID');
        return;
    }
    
    $db = new Database();
    $result = $db->removeAdmin($userId);
    
    if ($result) {
        // Log activity
        Database::logActivity('admin_remove', 'ادمین حذف شد', [
            'user_id' => $userId
        ]);
        
        sendResponse(true, [], 'Admin removed successfully');
    } else {
        sendResponse(false, [], 'Failed to remove admin or admin not found');
    }
}

function handleGetWebhookInfo() {
    $result = Telegram::request('getWebhookInfo');
    
    if ($result && isset($result['ok']) && $result['ok']) {
        sendResponse(true, $result['result']);
    } else {
        sendResponse(false, [], 'Failed to get webhook info');
    }
}

function handleSetWebhook($request) {
    $customUrl = $request['webhook_url'] ?? '';
    
    // اگر URL سفارشی نباشد، از URL فعلی سرور استفاده کن
    if (empty($customUrl)) {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $scriptPath = dirname($_SERVER['SCRIPT_NAME']);
        $customUrl = $protocol . '://' . $host . $scriptPath . '/index.php';
    }
    
    $result = Telegram::request('setWebhook', [
        'url' => $customUrl,
        'max_connections' => 100,
        'allowed_updates' => json_encode(['message', 'callback_query'])
    ]);
    
    if ($result && isset($result['ok']) && $result['ok']) {
        sendResponse(true, [], 'Webhook set successfully');
    } else {
        $errorMsg = $result['description'] ?? 'Failed to set webhook';
        sendResponse(false, [], $errorMsg);
    }
}

function handleDeleteWebhook() {
    $result = Telegram::request('deleteWebhook');
    
    if ($result && isset($result['ok']) && $result['ok']) {
        sendResponse(true, [], 'Webhook deleted successfully');
    } else {
        sendResponse(false, [], 'Failed to delete webhook');
    }
}

// Force Join Channels Management
function handleGetForceJoinChannels() {
    $settings = Database::getSettings();
    $channels = $settings['force_join_channels'] ?? [];
    sendResponse(true, ['channels' => $channels]);
}

function handleCheckChannelInfo($request) {
    $username = $request['username'] ?? '';
    
    if (empty($username)) {
        sendResponse(false, [], 'Username is required');
        return;
    }
    
    // Remove @ if present
    $username = ltrim($username, '@');
    $chatId = '@' . $username;
    
    // Get channel info
    $chatInfo = Telegram::request('getChat', [
        'chat_id' => $chatId
    ]);
    
    if (!$chatInfo || !isset($chatInfo['ok']) || !$chatInfo['ok']) {
        sendResponse(false, [], 'کانال یافت نشد یا ربات دسترسی ندارد');
        return;
    }
    
    // Check if bot is admin
    $adminsResult = Telegram::request('getChatAdministrators', [
        'chat_id' => $chatId
    ]);
    
    if (!$adminsResult || !isset($adminsResult['ok']) || !$adminsResult['ok']) {
        sendResponse(false, [], 'خطا در دریافت لیست ادمین‌ها');
        return;
    }
    
    $botId = BOT_ID;
    $isAdmin = false;
    
    foreach ($adminsResult['result'] as $admin) {
        if ($admin['user']['id'] == $botId) {
            $isAdmin = true;
            break;
        }
    }
    
    if (!$isAdmin) {
        sendResponse(false, [], 'ربات ادمین این کانال نیست');
        return;
    }
    
    // Return channel info
    $channelData = [
        'username' => $username,
        'title' => $chatInfo['result']['title'] ?? '',
        'type' => $chatInfo['result']['type'] ?? '',
        'members_count' => $chatInfo['result']['members_count'] ?? 0
    ];
    
    sendResponse(true, $channelData, 'اطلاعات کانال دریافت شد');
}

function handleAddForceJoinChannel($request) {
    $username = $request['username'] ?? '';
    $title = $request['title'] ?? '';
    
    if (empty($username)) {
        sendResponse(false, [], 'Username is required');
        return;
    }
    
    if (empty($title)) {
        sendResponse(false, [], 'Title is required');
        return;
    }
    
    // Remove @ if present
    $username = ltrim($username, '@');
    
    $settings = Database::getSettings();
    
    if (!isset($settings['force_join_channels'])) {
        $settings['force_join_channels'] = [];
    }
    
    // Check if channel already exists
    foreach ($settings['force_join_channels'] as $channel) {
        if ($channel['username'] === $username) {
            sendResponse(false, [], 'این کانال قبلا اضافه شده است');
            return;
        }
    }
    
    // Add new channel
    $settings['force_join_channels'][] = [
        'username' => $username,
        'title' => $title
    ];
    
    if (Database::saveSettings($settings)) {
        // Log activity
        Database::logActivity('channel_add', 'کانال جدید اضافه شد', [
            'username' => $username,
            'title' => $title
        ]);
        
        sendResponse(true, [], 'Channel added successfully');
    } else {
        sendResponse(false, [], 'Failed to save channel');
    }
}

function handleUpdateForceJoinChannel($request) {
    $username = $request['username'] ?? '';
    $title = $request['title'] ?? '';
    
    if (empty($username)) {
        sendResponse(false, [], 'Username is required');
        return;
    }
    
    $settings = Database::getSettings();
    
    if (!isset($settings['force_join_channels'])) {
        sendResponse(false, [], 'No channels found');
        return;
    }
    
    $found = false;
    foreach ($settings['force_join_channels'] as &$channel) {
        if ($channel['username'] === $username) {
            $channel['title'] = $title;
            $found = true;
            break;
        }
    }
    
    if (!$found) {
        sendResponse(false, [], 'Channel not found');
        return;
    }
    
    if (Database::saveSettings($settings)) {
        sendResponse(true, [], 'Channel updated successfully');
    } else {
        sendResponse(false, [], 'Failed to update channel');
    }
}

function handleDeleteForceJoinChannel($request) {
    $username = $request['username'] ?? '';
    
    if (empty($username)) {
        sendResponse(false, [], 'Username is required');
        return;
    }
    
    $settings = Database::getSettings();
    
    if (!isset($settings['force_join_channels'])) {
        sendResponse(false, [], 'No channels found');
        return;
    }
    
    $newChannels = array_filter($settings['force_join_channels'], function($channel) use ($username) {
        return $channel['username'] !== $username;
    });
    
    // Reset array keys
    $settings['force_join_channels'] = array_values($newChannels);
    
    if (Database::saveSettings($settings)) {
        // Log activity
        Database::logActivity('channel_delete', 'کانال حذف شد', [
            'username' => $username
        ]);
        
        sendResponse(true, [], 'Channel deleted successfully');
    } else {
        sendResponse(false, [], 'Failed to delete channel');
    }
}

/**
 * Check user type (admin or regular user)
 */
function handleCheckUserType($request) {
    // Get user_id from request
    $userId = $request['user_id'] ?? null;
    
    if (!$userId) {
        sendResponse(false, [], 'User ID is required');
        return;
    }
    
    // Check if user is admin
    $adminsFile = __DIR__ . '/data/admin_tokens.json';
    
    if (!file_exists($adminsFile)) {
        // No admins file, user is regular user
        sendResponse(true, [
            'user_type' => 'user',
            'user_id' => $userId
        ]);
        return;
    }
    
    $admins = json_decode(file_get_contents($adminsFile), true) ?? [];
    
    // Check if user_id exists in admins
    $isAdmin = false;
    foreach ($admins as $admin) {
        if (isset($admin['user_id']) && $admin['user_id'] == $userId) {
            $isAdmin = true;
            break;
        }
    }
    
    sendResponse(true, [
        'user_type' => $isAdmin ? 'admin' : 'user',
        'user_id' => $userId
    ]);
}

