// Admin Panel JavaScript

// Initialize Telegram Web App
const tg = window.Telegram?.WebApp;
if (tg) {
    tg.expand();
    tg.ready();
}

// API Configuration
const API_URL = window.location.origin + '/admin_api.php';

// State
let currentTab = 'stats';
let adminData = null;
let usersData = [];
let filteredUsers = [];

// Dark Mode Functions
function initDarkMode() {
    const savedTheme = localStorage.getItem('theme');
    
    // پیشفرض دارک مود
    if (savedTheme === 'light') {
        // فقط اگر کاربر صریحاً لایت مود رو انتخاب کرده
        document.body.classList.remove('dark-mode');
        updateThemeIcon(false);
    } else {
        // پیشفرض یا وقتی دارک مود ذخیره شده
        document.body.classList.add('dark-mode');
        updateThemeIcon(true);
    }
}

function toggleDarkMode() {
    const isDark = document.body.classList.toggle('dark-mode');
    localStorage.setItem('theme', isDark ? 'dark' : 'light');
    updateThemeIcon(isDark);
    
    // Show notification
    showNotification(isDark ? 'حالت تاریک فعال شد' : 'حالت روشن فعال شد', 'success');
}

function updateThemeIcon(isDark) {
    const icon = document.getElementById('themeIcon');
    if (icon) {
        icon.className = isDark ? 'fas fa-sun' : 'fas fa-moon';
    }
}

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    // Initialize dark mode
    initDarkMode();
    
    // Check if opened from Telegram
    const telegramUser = tg?.initDataUnsafe?.user;
    if (telegramUser) {
        // Try to login with Telegram data
        loginWithTelegram(telegramUser);
    } else {
        // Check if has saved token
        const token = localStorage.getItem('admin_token');
        if (token) {
            checkAuth();
        } else {
            // Show modal: not opened from Telegram
            showTelegramNotLoadedModal();
        }
    }
});

// Check Authentication
function checkAuth() {
    const token = localStorage.getItem('admin_token');
    
    if (!token) {
        showLoginScreen();
    } else {
        // Verify token with API
        verifyToken(token);
    }
}

async function verifyToken(token) {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'verifyToken',
                token: token
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            adminData = result.data;
            showDashboard();
            loadStats();
        } else {
            localStorage.removeItem('admin_token');
            showTelegramNotLoadedModal();
        }
    } catch (error) {
        console.error('Token verification error:', error);
        showTelegramNotLoadedModal();
    }
}

function showTelegramNotLoadedModal() {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('dashboard').style.display = 'none';
    document.getElementById('telegramNotLoadedModal').style.display = 'flex';
}

function showDashboard() {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('telegramNotLoadedModal').style.display = 'none';
    document.getElementById('dashboard').style.display = 'block';
    
    if (adminData) {
        // Update admin info card
        const firstName = adminData.first_name || 'ادمین';
        const lastName = adminData.last_name || '';
        const username = adminData.username || 'admin';
        
        // Set avatar text (first letter of first name)
        const avatarText = firstName.charAt(0).toUpperCase();
        document.getElementById('adminAvatarText').textContent = avatarText;
        
        // Set admin name
        const fullName = lastName ? `${firstName} ${lastName}` : firstName;
        document.getElementById('adminNameDisplay').textContent = fullName;
        
        // Set username
        document.getElementById('adminUsername').textContent = `@${username}`;
        
        // Keep old element for compatibility
        if (document.getElementById('adminName')) {
            document.getElementById('adminName').textContent = firstName;
        }
    }
}

// Login with Telegram
async function loginWithTelegram(telegramUser) {
    try {
        console.log('Logging in with Telegram user:', telegramUser);
        console.log('API URL:', API_URL);
        
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'loginWithTelegram',
                telegram_data: {
                    id: telegramUser.id,
                    first_name: telegramUser.first_name,
                    last_name: telegramUser.last_name,
                    username: telegramUser.username
                }
            })
        });
        
        console.log('Response status:', response.status);
        const text = await response.text();
        console.log('Response text:', text.substring(0, 200));
        
        let result;
        try {
            result = JSON.parse(text);
        } catch (e) {
            console.error('JSON parse error:', e);
            console.error('Response was:', text.substring(0, 500));
            throw new Error('پاسخ سرور نامعتبر است');
        }
        
        if (result.success) {
            localStorage.setItem('admin_token', result.token);
            adminData = result.data;
            // Hide modal and show dashboard
            document.getElementById('telegramNotLoadedModal').style.display = 'none';
            showDashboard();
            loadStats();
            showNotification('خوش آمدید! ورود از طریق تلگرام موفق بود', 'success');
        } else {
            showTelegramNotLoadedModal();
            showNotification(result.message || 'شما دسترسی ادمین ندارید', 'error');
        }
    } catch (error) {
        console.error('Telegram login error:', error);
        showTelegramNotLoadedModal();
        showNotification(error.message || 'خطا در ارتباط با سرور', 'error');
    }
}

// Login with Test User
function loginWithTestUser() {
    const testAdmin = {
        id: 987654321,
        first_name: 'ادمین',
        last_name: 'تست',
        username: 'test_admin'
    };
    
    loginWithTelegram(testAdmin);
}

// Handle Login
// Handle Logout
function handleLogout() {
    localStorage.removeItem('admin_token');
    location.reload();
}

// Switch Tab
function switchTab(tabName) {
    currentTab = tabName;
    
    // Update tab buttons
    document.querySelectorAll('.nav-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    document.querySelector(`.nav-tab[data-tab="${tabName}"]`).classList.add('active');
    
    // Update content
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.remove('active');
    });
    document.getElementById(`${tabName}Tab`).classList.add('active');
    
    // Load data for tab
    switch(tabName) {
        case 'stats':
            loadStats();
            break;
        case 'users':
            loadUsers();
            break;
        case 'files':
            loadFiles();
            break;
        case 'broadcast':
            loadUsers(); // Load users for broadcast stats
            break;
        case 'appeals':
            loadAppeals();
            break;
        case 'admins':
            loadAdmins();
            break;
        case 'settings':
            loadWebhookInfo();
            loadForceJoinChannels();
            break;
    }
}

// Load Stats
async function loadStats() {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getStats',
                token: localStorage.getItem('admin_token')
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            const stats = result.data;
            
            document.getElementById('totalUsers').textContent = stats.total_users || 0;
            document.getElementById('activeUsers').textContent = stats.active_users || 0;
            document.getElementById('blockedUsers').textContent = stats.blocked_users || 0;
            document.getElementById('totalFiles').textContent = stats.total_files || 0;
            document.getElementById('totalDownloads').textContent = stats.total_downloads || 0;
            document.getElementById('totalWarnings').textContent = stats.total_warnings || 0;
            
            renderRecentActivity(stats.recent_activity || []);
        }
    } catch (error) {
        console.error('Error loading stats:', error);
    }
}

function renderRecentActivity(activities) {
    const container = document.getElementById('recentActivity');
    
    if (!activities || activities.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #8E8E93; padding: 20px;">فعالیت اخیری وجود ندارد</p>';
        return;
    }
    
    container.innerHTML = activities.map(activity => `
        <div class="activity-item">
            <div class="activity-icon" style="background: ${getActivityColor(activity.type)}">
                <i class="${getActivityIcon(activity.type)}"></i>
            </div>
            <div class="activity-info">
                <div class="activity-title">${activity.title}</div>
                <div class="activity-time">${formatTime(activity.time)}</div>
            </div>
        </div>
    `).join('');
}

function getActivityColor(type) {
    const colors = {
        'user_join': 'linear-gradient(135deg, #34C759, #30B350)',
        'user_block': 'linear-gradient(135deg, #FF3B30, #CC2E26)',
        'user_unblock': 'linear-gradient(135deg, #34C759, #30B350)',
        'file_upload': 'linear-gradient(135deg, #0A84FF, #0066CC)',
        'warning': 'linear-gradient(135deg, #FF9500, #CC7700)',
        'appeal': 'linear-gradient(135deg, #FF9F0A, #CC7F08)',
        'channel_add': 'linear-gradient(135deg, #5E5CE6, #4A4ACB)',
        'channel_delete': 'linear-gradient(135deg, #FF3B30, #CC2E26)',
        'admin_add': 'linear-gradient(135deg, #34C759, #30B350)',
        'admin_remove': 'linear-gradient(135deg, #FF3B30, #CC2E26)'
    };
    return colors[type] || 'linear-gradient(135deg, #8E8E93, #636366)';
}

function getActivityIcon(type) {
    const icons = {
        'user_join': 'fas fa-user-plus',
        'user_block': 'fas fa-user-slash',
        'user_unblock': 'fas fa-user-check',
        'file_upload': 'fas fa-upload',
        'warning': 'fas fa-exclamation-triangle',
        'appeal': 'fas fa-gavel',
        'channel_add': 'fas fa-plus-circle',
        'channel_delete': 'fas fa-trash',
        'admin_add': 'fas fa-user-shield',
        'admin_remove': 'fas fa-user-times'
    };
    return icons[type] || 'fas fa-circle';
}

// Load Users
async function loadUsers() {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getUsers',
                token: localStorage.getItem('admin_token')
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            usersData = result.data || [];
            filteredUsers = usersData;
            renderUsers();
            
            // Update broadcast tab if it's active
            if (currentTab === 'broadcast') {
                updateBroadcastStats();
            }
        }
    } catch (error) {
        console.error('Error loading users:', error);
    }
}

function updateBroadcastStats() {
    // This will be used by broadcast confirm modal
    // No need to display anything yet, just keeps data fresh
}

function renderUsers() {
    const container = document.getElementById('usersList');
    
    if (filteredUsers.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #8E8E93; padding: 20px;">کاربری یافت نشد</p>';
        return;
    }
    
    container.innerHTML = filteredUsers.map(user => `
        <div class="user-item" onclick="showUserDetails(${user.user_id})">
            <div class="user-avatar">
                ${user.first_name ? user.first_name.charAt(0) : 'U'}
            </div>
            <div class="user-info">
                <div class="user-name">${user.first_name || 'کاربر'} ${user.last_name || ''}</div>
                <div class="user-meta">
                    <span><i class="fas fa-id-card"></i> ${user.user_id}</span>
                    ${user.username ? `<span><i class="fas fa-at"></i> ${user.username}</span>` : ''}
                    <span class="badge ${user.is_blocked ? 'blocked' : 'active'}">
                        ${user.is_blocked ? 'مسدود' : 'فعال'}
                    </span>
                </div>
            </div>
            <div class="user-actions" onclick="event.stopPropagation()">
                <button class="action-btn view" onclick="showUserDetails(${user.user_id})" title="مشاهده جزئیات">
                    <i class="fas fa-eye"></i>
                </button>
                ${user.is_blocked ? 
                    `<button class="action-btn unblock" onclick="unblockUser(${user.user_id})" title="رفع مسدودیت">
                        <i class="fas fa-unlock"></i>
                    </button>` :
                    `<button class="action-btn block" onclick="blockUser(${user.user_id})" title="مسدود کردن">
                        <i class="fas fa-ban"></i>
                    </button>`
                }
            </div>
        </div>
    `).join('');
}

function searchUsers() {
    const query = document.getElementById('userSearch').value.toLowerCase();
    
    if (!query) {
        filteredUsers = usersData;
    } else {
        filteredUsers = usersData.filter(user => {
            return (
                (user.first_name && user.first_name.toLowerCase().includes(query)) ||
                (user.last_name && user.last_name.toLowerCase().includes(query)) ||
                (user.username && user.username.toLowerCase().includes(query)) ||
                user.user_id.toString().includes(query)
            );
        });
    }
    
    renderUsers();
}

// Show User Details
async function showUserDetails(userId) {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getUserDetails',
                token: localStorage.getItem('admin_token'),
                user_id: userId
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            const user = result.data;
            const modalBody = document.getElementById('userModalBody');
            
            modalBody.innerHTML = `
                <div style="display: flex; flex-direction: column; gap: 16px;">
                    <div class="input-group">
                        <label><i class="fas fa-user"></i> نام و نام خانوادگی</label>
                        <input type="text" value="${user.first_name || ''} ${user.last_name || ''}" readonly>
                    </div>
                    <div class="input-group">
                        <label><i class="fas fa-at"></i> نام کاربری</label>
                        <input type="text" value="${user.username || 'ندارد'}" readonly>
                    </div>
                    <div class="input-group">
                        <label><i class="fas fa-id-card"></i> شناسه کاربری</label>
                        <input type="text" value="${user.user_id}" readonly>
                    </div>
                    <div class="input-group">
                        <label><i class="fas fa-calendar"></i> تاریخ عضویت</label>
                        <input type="text" value="${new Date(user.join_date).toLocaleDateString('fa-IR')}" readonly>
                    </div>
                    <div class="input-group">
                        <label><i class="fas fa-download"></i> تعداد دانلودها</label>
                        <input type="text" value="${user.download_count || 0}" readonly>
                    </div>
                    <div class="input-group">
                        <label><i class="fas fa-exclamation-triangle"></i> تعداد اخطارها</label>
                        <input type="text" value="${user.warning_count || 0}" readonly>
                    </div>
                    <div class="input-group">
                        <label><i class="fas fa-info-circle"></i> وضعیت</label>
                        <input type="text" value="${user.is_blocked ? 'مسدود شده' : 'فعال'}" readonly style="color: ${user.is_blocked ? '#FF3B30' : '#34C759'}; font-weight: 600;">
                    </div>
                    
                    ${user.warnings && user.warnings.length > 0 ? `
                        <div style="margin-top: 16px;">
                            <h4 style="margin-bottom: 12px;"><i class="fas fa-list"></i> اخطارهای کاربر</h4>
                            ${user.warnings.map((warning, index) => `
                                <div style="padding: 12px; background: rgba(255, 149, 0, 0.1); border-radius: 12px; margin-bottom: 8px; border-right: 3px solid #FF9500; display: flex; justify-content: space-between; align-items: flex-start; gap: 12px;">
                                    <div style="flex: 1;">
                                        <div style="font-weight: 600; margin-bottom: 4px;">${warning.reason}</div>
                                        <div style="font-size: 13px; color: #8E8E93;">${new Date(warning.date).toLocaleDateString('fa-IR')}</div>
                                    </div>
                                    <button onclick="deleteWarning(${user.user_id}, ${index})" style="background: rgba(255, 59, 48, 0.1); color: #FF3B30; border: none; width: 32px; height: 32px; border-radius: 8px; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.2s;" onmouseover="this.style.background='rgba(255, 59, 48, 0.2)'" onmouseout="this.style.background='rgba(255, 59, 48, 0.1)'" title="حذف اخطار">
                                        <i class="fas fa-trash" style="font-size: 14px;"></i>
                                    </button>
                                </div>
                            `).join('')}
                        </div>
                    ` : ''}
                    
                    <div style="display: flex; gap: 12px; margin-top: 16px;">
                        ${!user.is_blocked ? 
                            `<button class="primary-btn" onclick="sendWarning(${user.user_id})" style="flex: 1; background: linear-gradient(135deg, #FF9500, #CC7700);">
                                <i class="fas fa-exclamation-triangle"></i> ارسال اخطار
                            </button>
                            <button class="primary-btn" onclick="blockUser(${user.user_id})" style="flex: 1; background: linear-gradient(135deg, #FF3B30, #CC2E26);">
                                <i class="fas fa-ban"></i> مسدود کردن
                            </button>` :
                            `<button class="primary-btn" onclick="unblockUser(${user.user_id})" style="flex: 1; background: linear-gradient(135deg, #34C759, #30B350);">
                                <i class="fas fa-unlock"></i> رفع مسدودیت
                            </button>`
                        }
                    </div>
                </div>
            `;
            
            document.getElementById('userModal').style.display = 'flex';
        }
    } catch (error) {
        console.error('Error loading user details:', error);
        showNotification('خطا در بارگذاری اطلاعات کاربر', 'error');
    }
}

function closeUserModal() {
    document.getElementById('userModal').style.display = 'none';
}

// Block User
async function blockUser(userId) {
    showConfirmModal(
        'مسدود کردن کاربر',
        'آیا از مسدود کردن این کاربر اطمینان دارید؟ کاربر تا زمان رفع مسدودیت نمی‌تواند از ربات استفاده کند.',
        async () => {
            try {
                const response = await fetch(API_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'blockUser',
                        token: localStorage.getItem('admin_token'),
                        user_id: userId
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('کاربر با موفقیت مسدود شد', 'success');
                    closeUserModal();
                    loadUsers();
                    loadStats();
                } else {
                    showNotification('خطا در مسدود کردن کاربر', 'error');
                }
            } catch (error) {
                console.error('Error blocking user:', error);
                showNotification('خطا در مسدود کردن کاربر', 'error');
            }
        }
    );
}

// Unblock User
async function unblockUser(userId) {
    showConfirmModal(
        'رفع مسدودیت کاربر',
        'آیا از رفع مسدودیت این کاربر اطمینان دارید؟',
        async () => {
            try {
                const response = await fetch(API_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'unblockUser',
                        token: localStorage.getItem('admin_token'),
                        user_id: userId
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('مسدودیت کاربر با موفقیت رفع شد', 'success');
                    closeUserModal();
                    loadUsers();
                    loadStats();
                } else {
                    showNotification('خطا در رفع مسدودیت', 'error');
                }
            } catch (error) {
                console.error('Error unblocking user:', error);
                showNotification('خطا در رفع مسدودیت', 'error');
            }
        }
    );
}

// Send Warning
let warningUserId = null;

async function sendWarning(userId) {
    warningUserId = userId;
    document.getElementById('warningReason').value = '';
    document.getElementById('warningModal').style.display = 'flex';
}

function closeWarningModal() {
    document.getElementById('warningModal').style.display = 'none';
    warningUserId = null;
}

async function confirmSendWarning() {
    const reason = document.getElementById('warningReason').value.trim();
    
    if (!reason) {
        showNotification('لطفا دلیل اخطار را وارد کنید', 'error');
        return;
    }
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'sendWarning',
                token: localStorage.getItem('admin_token'),
                user_id: warningUserId,
                reason: reason
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification('اخطار با موفقیت ارسال شد', 'success');
            closeWarningModal();
            closeUserModal();
            loadStats();
        } else {
            showNotification('خطا در ارسال اخطار', 'error');
        }
    } catch (error) {
        console.error('Error sending warning:', error);
        showNotification('خطا در ارسال اخطار', 'error');
    }
}

// Delete Warning
async function deleteWarning(userId, warningIndex) {
    showConfirmModal(
        'حذف اخطار',
        'آیا از حذف این اخطار اطمینان دارید؟',
        async () => {
            try {
                const response = await fetch(API_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'deleteWarning',
                        token: localStorage.getItem('admin_token'),
                        user_id: userId,
                        warning_index: warningIndex
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('اخطار با موفقیت حذف شد', 'success');
                    showUserDetails(userId);
                    loadStats();
                } else {
                    showNotification('خطا در حذف اخطار', 'error');
                }
            } catch (error) {
                console.error('Error deleting warning:', error);
                showNotification('خطا در حذف اخطار', 'error');
            }
        }
    );
}

// Load Files
async function loadFiles() {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getFiles',
                token: localStorage.getItem('admin_token')
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            renderFiles(result.data || []);
        }
    } catch (error) {
        console.error('Error loading files:', error);
    }
}

function renderFiles(files) {
    const container = document.getElementById('filesList');
    
    if (files.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #8E8E93; padding: 20px; grid-column: 1 / -1;">فایلی یافت نشد</p>';
        return;
    }
    
    container.innerHTML = files.map(file => `
        <div class="file-card" onclick='showFileDetails(${JSON.stringify(file).replace(/'/g, "&apos;")})' style="cursor: pointer;">
            <div class="file-card-icon">
                <i class="${getFileIcon(file.file_type)}"></i>
            </div>
            <div class="file-card-name" title="${file.file_name}">${file.file_name}</div>
            <div class="file-card-size">${formatFileSize(file.file_size)}</div>
        </div>
    `).join('');
}

function getFileIcon(type) {
    const icons = {
        'document': 'fas fa-file-alt',
        'video': 'fas fa-video',
        'audio': 'fas fa-music',
        'photo': 'fas fa-image',
        'archive': 'fas fa-file-archive'
    };
    return icons[type] || 'fas fa-file';
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

// Send Broadcast
let broadcastData = null;

async function sendBroadcast() {
    const message = document.getElementById('broadcastMessage').value.trim();
    const sendToAll = document.getElementById('sendToAll').checked;
    const sendToActive = document.getElementById('sendToActive').checked;
    
    if (!message) {
        showNotification('لطفاً متن پیام را وارد کنید', 'error');
        return;
    }
    
    // ذخیره داده‌های پیام برای ارسال بعد از تایید
    broadcastData = {
        message: message,
        sendToAll: sendToAll,
        sendToActive: sendToActive
    };
    
    // نمایش مودال تایید
    showBroadcastConfirmModal(message, sendToAll, sendToActive);
}

function showBroadcastConfirmModal(message, sendToAll, sendToActive) {
    // محاسبه تعداد کاربران
    const totalUsers = usersData.length;
    const activeUsers = usersData.filter(u => !u.is_blocked).length;
    const targetCount = sendToActive ? activeUsers : totalUsers;
    
    // تنظیم اطلاعات مودال
    document.getElementById('broadcastUserCount').textContent = targetCount.toLocaleString('fa-IR');
    document.getElementById('broadcastFilter').textContent = sendToActive ? 'فقط کاربران فعال' : 'همه کاربران';
    document.getElementById('broadcastMessageLength').textContent = message.length.toLocaleString('fa-IR');
    
    // نمایش مودال
    const modal = document.getElementById('broadcastConfirmModal');
    modal.style.display = 'flex';
}

function closeBroadcastConfirmModal() {
    const modal = document.getElementById('broadcastConfirmModal');
    modal.style.display = 'none';
    broadcastData = null;
}

async function confirmBroadcastSend() {
    if (!broadcastData) return;
    
    // ذخیره داده‌ها قبل از بستن مودال
    const data = {
        message: broadcastData.message,
        sendToAll: broadcastData.sendToAll,
        sendToActive: broadcastData.sendToActive
    };
    
    closeBroadcastConfirmModal();
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'sendBroadcast',
                token: localStorage.getItem('admin_token'),
                message: data.message,
                send_to_all: data.sendToAll,
                send_to_active: data.sendToActive
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification(`پیام با موفقیت برای ${result.sent_count} کاربر ارسال شد`, 'success');
            document.getElementById('broadcastMessage').value = '';
        } else {
            showNotification('خطا در ارسال پیام', 'error');
        }
    } catch (error) {
        console.error('Error sending broadcast:', error);
        showNotification('خطا در ارسال پیام', 'error');
    }
}

// Load Appeals
async function loadAppeals() {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getAppeals',
                token: localStorage.getItem('admin_token')
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            renderAppeals(result.data || []);
        }
    } catch (error) {
        console.error('Error loading appeals:', error);
    }
}

function renderAppeals(appeals) {
    const container = document.getElementById('appealsList');
    
    if (appeals.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #8E8E93; padding: 20px;">اعتراضی یافت نشد</p>';
        return;
    }
    
    container.innerHTML = appeals.map(appeal => `
        <div class="appeal-item">
            <div class="appeal-header">
                <div class="appeal-user">
                    ${appeal.first_name || 'کاربر'} ${appeal.last_name || ''}
                    ${appeal.username ? `(@${appeal.username})` : ''}
                    <span style="display: block; font-size: 13px; color: #8E8E93; margin-top: 4px;">
                        <i class="fas fa-id-card"></i> شناسه: ${appeal.user_id}
                    </span>
                </div>
                <div class="appeal-date">${formatTime(appeal.timestamp)}</div>
            </div>
            <div class="appeal-text">${appeal.appeal_text}</div>
            <div class="appeal-actions">
                <button class="primary-btn" onclick="acceptAppeal('${appeal.id}', ${appeal.user_id})">
                    <i class="fas fa-check"></i> قبول و رفع مسدودیت
                </button>
                <button class="primary-btn" onclick="rejectAppeal('${appeal.id}', ${appeal.user_id})" style="background: linear-gradient(135deg, #FF3B30, #CC2E26);">
                    <i class="fas fa-times"></i> رد کردن
                </button>
            </div>
        </div>
    `).join('');
}

async function acceptAppeal(appealId, userId) {
    showConfirmModal(
        'قبول اعتراض',
        'آیا از قبول اعتراض و رفع مسدودیت کاربر اطمینان دارید؟ حساب کاربر رفع مسدودیت خواهد شد و پیام تایید برای او ارسال می‌شود.',
        async () => {
            try {
                const response = await fetch(API_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'acceptAppeal',
                        token: localStorage.getItem('admin_token'),
                        appeal_id: appealId,
                        user_id: userId
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('اعتراض قبول شد، مسدودیت رفع شد و پیام به کاربر ارسال شد', 'success');
                    loadAppeals();
                    loadStats();
                } else {
                    showNotification('خطا در پردازش اعتراض', 'error');
                }
            } catch (error) {
                console.error('Error accepting appeal:', error);
                showNotification('خطا در پردازش اعتراض', 'error');
            }
        }
    );
}

// Reject Appeal with Modal
let pendingRejectAppeal = null;

async function rejectAppeal(appealId, userId) {
    // Store data for later use
    pendingRejectAppeal = { appealId, userId };
    
    // Show reject reason modal
    const modal = document.getElementById('rejectReasonModal');
    const textarea = document.getElementById('rejectReasonText');
    textarea.value = '';
    modal.style.display = 'flex';
    setTimeout(() => textarea.focus(), 100);
}

function closeRejectReasonModal() {
    const modal = document.getElementById('rejectReasonModal');
    modal.style.display = 'none';
    pendingRejectAppeal = null;
}

async function submitRejectReason() {
    const textarea = document.getElementById('rejectReasonText');
    const reason = textarea.value.trim();
    
    if (!reason) {
        showNotification('لطفاً دلیل رد را وارد کنید', 'error');
        return;
    }
    
    if (!pendingRejectAppeal) {
        showNotification('خطا: اطلاعات اعتراض یافت نشد', 'error');
        return;
    }
    
    const { appealId, userId } = pendingRejectAppeal;
    
    // Close modal first
    closeRejectReasonModal();
    
    // Show confirmation
    showConfirmModal(
        'رد اعتراض',
        'آیا از رد کردن اعتراض کاربر اطمینان دارید؟ پیام رد به کاربر ارسال خواهد شد.',
        async () => {
            try {
                const response = await fetch(API_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'rejectAppeal',
                        token: localStorage.getItem('admin_token'),
                        appeal_id: appealId,
                        user_id: userId,
                        reject_reason: reason
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('اعتراض رد شد و پیام به کاربر ارسال شد', 'success');
                    loadAppeals();
                    loadStats();
                } else {
                    showNotification('خطا در پردازش اعتراض', 'error');
                }
            } catch (error) {
                console.error('Error rejecting appeal:', error);
                showNotification('خطا در پردازش اعتراض', 'error');
            }
        }
    );
}

// Load Admins
async function loadAdmins() {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getAdmins',
                token: localStorage.getItem('admin_token')
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            renderAdmins(result.data || []);
        }
    } catch (error) {
        console.error('Error loading admins:', error);
    }
}

function renderAdmins(admins) {
    const container = document.getElementById('adminsList');
    
    if (admins.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: var(--text-secondary); padding: 20px;">ادمینی یافت نشد</p>';
        return;
    }
    
    container.innerHTML = admins.map(admin => `
        <div class="admin-item">
            <div class="admin-info">
                <div class="user-avatar">
                    ${admin.first_name ? admin.first_name.charAt(0) : 'A'}
                </div>
                <div class="admin-details">
                    <div class="admin-name">${admin.first_name || 'ادمین'}</div>
                    <div class="admin-id">ID: ${admin.user_id}</div>
                </div>
                ${admin.is_main ? '<div class="admin-badge">ادمین اصلی</div>' : ''}
            </div>
            ${!admin.is_main ? `
                <button class="action-btn delete" onclick="confirmRemoveAdmin('${admin.user_id}', '${admin.first_name || 'ادمین'}')">
                    <i class="fas fa-trash"></i>
                </button>
            ` : ''}
        </div>
    `).join('');
}

function showAddAdminModal() {
    document.getElementById('addAdminModal').style.display = 'flex';
}

function closeAddAdminModal() {
    document.getElementById('addAdminModal').style.display = 'none';
    document.getElementById('newAdminId').value = '';
}

async function addNewAdmin() {
    const userId = document.getElementById('newAdminId').value.trim();
    
    if (!userId) {
        showNotification('لطفاً شناسه کاربری را وارد کنید', 'error');
        return;
    }
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'addAdmin',
                token: localStorage.getItem('admin_token'),
                user_id: parseInt(userId)
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification('ادمین با موفقیت اضافه شد', 'success');
            closeAddAdminModal();
            loadAdmins();
        } else {
            showNotification(result.message || 'خطا در افزودن ادمین', 'error');
        }
    } catch (error) {
        console.error('Error adding admin:', error);
        showNotification('خطا در افزودن ادمین', 'error');
    }
}

function confirmRemoveAdmin(userId, name) {
    showConfirmModal(
        'حذف ادمین',
        `آیا از حذف ${name} اطمینان دارید؟ این عملیات غیرقابل بازگشت است.`,
        () => removeAdmin(userId),
        'حذف',
        'danger'
    );
}

async function removeAdmin(userId) {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'removeAdmin',
                token: localStorage.getItem('admin_token'),
                user_id: parseInt(userId)
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification('ادمین با موفقیت حذف شد', 'success');
            loadAdmins();
        } else {
            showNotification(result.message || 'خطا در حذف ادمین', 'error');
        }
    } catch (error) {
        console.error('Error removing admin:', error);
        showNotification('خطا در حذف ادمین', 'error');
    }
}

// File Details Modal Functions
let currentFileId = null;
let currentFileData = null;

function showFileDetails(fileData) {
    currentFileId = fileData.id;
    currentFileData = fileData;
    
    document.getElementById('fileDetailName').textContent = fileData.file_name || '-';
    document.getElementById('fileDetailType').textContent = fileData.file_type || '-';
    document.getElementById('fileDetailSize').textContent = formatFileSize(fileData.file_size || 0);
    document.getElementById('fileDetailUploader').textContent = fileData.uploaded_by || '-';
    document.getElementById('fileDetailDate').textContent = fileData.upload_date || '-';
    document.getElementById('fileDetailDownloads').textContent = fileData.download_count || '0';
    
    const isActive = fileData.is_active !== false;
    const statusText = isActive ? '✅ فعال' : '❌ غیرفعال';
    const statusColor = isActive ? '#34C759' : '#FF3B30';
    document.getElementById('fileDetailStatus').innerHTML = `<span style="color: ${statusColor}">${statusText}</span>`;
    
    document.getElementById('fileDetailsModal').style.display = 'flex';
}

function closeFileDetailsModal() {
    document.getElementById('fileDetailsModal').style.display = 'none';
    currentFileId = null;
    currentFileData = null;
}

function showEditFileModal() {
    if (!currentFileData) return;
    
    // Populate form
    document.getElementById('editFileTitle').value = currentFileData.title || '';
    document.getElementById('editFileCaption').value = currentFileData.caption || '';
    document.getElementById('editFilePassword').value = currentFileData.password || '';
    document.getElementById('editFileDownloadLimit').value = currentFileData.download_limit || 0;
    document.getElementById('editFileActive').checked = currentFileData.is_active !== false;
    
    // Set expire date if exists
    if (currentFileData.expire_date) {
        const date = new Date(currentFileData.expire_date);
        const localDate = new Date(date.getTime() - (date.getTimezoneOffset() * 60000)).toISOString().slice(0, 16);
        document.getElementById('editFileExpireDate').value = localDate;
    } else {
        document.getElementById('editFileExpireDate').value = '';
    }
    
    closeFileDetailsModal();
    document.getElementById('editFileModal').style.display = 'flex';
}

function closeEditFileModal() {
    document.getElementById('editFileModal').style.display = 'none';
}

async function saveFileEdits() {
    if (!currentFileId) return;
    
    const title = document.getElementById('editFileTitle').value.trim();
    const caption = document.getElementById('editFileCaption').value.trim();
    const password = document.getElementById('editFilePassword').value.trim();
    const downloadLimit = parseInt(document.getElementById('editFileDownloadLimit').value) || 0;
    const expireDate = document.getElementById('editFileExpireDate').value;
    const isActive = document.getElementById('editFileActive').checked;
    
    try {
        const updates = {
            title: title,
            caption: caption,
            is_active: isActive,
            download_limit: downloadLimit
        };
        
        if (password) {
            updates.password = password;
        } else {
            updates.password = '';
        }
        
        if (expireDate) {
            updates.expire_date = expireDate;
        } else {
            updates.expire_date = null;
        }
        
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'updateFile',
                token: localStorage.getItem('admin_token'),
                file_id: currentFileId,
                ...updates
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification('فایل با موفقیت ویرایش شد', 'success');
            closeEditFileModal();
            loadFiles();
        } else {
            showNotification('خطا در ویرایش فایل', 'error');
        }
    } catch (error) {
        console.error('Error updating file:', error);
        showNotification('خطا در ویرایش فایل', 'error');
    }
}

async function deleteFile() {
    if (!currentFileId) return;
    
    showConfirmModal(
        'حذف فایل',
        'آیا از حذف این فایل اطمینان دارید؟ این عملیات غیرقابل بازگشت است.',
        async () => {
            try {
                const response = await fetch(API_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'deleteFile',
                        token: localStorage.getItem('admin_token'),
                        file_id: currentFileId
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('فایل با موفقیت حذف شد', 'success');
                    closeFileDetailsModal();
                    loadFiles();
                } else {
                    showNotification('خطا در حذف فایل', 'error');
                }
            } catch (error) {
                console.error('Error deleting file:', error);
                showNotification('خطا در حذف فایل', 'error');
            }
        }
    );
}

// Utility Functions
function formatTime(timestamp) {
    const date = new Date(timestamp * 1000);
    const now = new Date();
    const diff = Math.floor((now - date) / 1000);
    
    if (diff < 60) return 'لحظاتی پیش';
    if (diff < 3600) return `${Math.floor(diff / 60)} دقیقه پیش`;
    if (diff < 86400) return `${Math.floor(diff / 3600)} ساعت پیش`;
    if (diff < 604800) return `${Math.floor(diff / 86400)} روز پیش`;
    
    return date.toLocaleDateString('fa-IR');
}

// Confirmation Modal
let confirmCallback = null;

function showConfirmModal(title, message, callback) {
    const modal = document.getElementById('confirmModal');
    const titleEl = document.getElementById('confirmTitle');
    const messageEl = document.getElementById('confirmMessage');
    
    titleEl.textContent = title;
    messageEl.textContent = message;
    confirmCallback = callback;
    
    modal.style.display = 'flex';
}

function closeConfirmModal() {
    const modal = document.getElementById('confirmModal');
    modal.style.display = 'none';
    confirmCallback = null;
}

async function confirmAction() {
    if (confirmCallback && typeof confirmCallback === 'function') {
        const callback = confirmCallback;
        closeConfirmModal();
        await callback();
    }
}

// Show Notification
function showNotification(message, type = 'success') {
    const toast = document.getElementById('notificationToast');
    const icon = toast.querySelector('i');
    const text = toast.querySelector('.notification-text');
    
    text.textContent = message;
    
    toast.className = 'notification-toast ' + type;
    if (type === 'success') {
        icon.className = 'fas fa-check-circle';
    } else if (type === 'error') {
        icon.className = 'fas fa-times-circle';
    }
    
    setTimeout(() => {
        toast.classList.add('show');
    }, 100);
    
    setTimeout(() => {
        toast.classList.remove('show');
    }, 3500);
}

// Force Join Channels Management Functions
async function loadForceJoinChannels() {
    const token = localStorage.getItem('admin_token');
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=getForceJoinChannels&token=${token}`
        });

        const data = await response.json();
        
        console.log('Force Join Channels Response:', data);
        
        if (data.success) {
            displayForceJoinChannels(data.data.channels || []);
        } else {
            showNotification('خطا در بارگذاری کانال‌ها', 'error');
        }
    } catch (error) {
        console.error('Error loading channels:', error);
        showNotification('خطا در اتصال به سرور', 'error');
    }
}

function displayForceJoinChannels(channels) {
    const channelsList = document.getElementById('channelsList');
    
    if (channels.length === 0) {
        channelsList.innerHTML = '<p style="text-align: center; color: var(--text-secondary); padding: 20px;">هیچ کانالی تعریف نشده است</p>';
        return;
    }
    
    channelsList.innerHTML = channels.map(channel => `
        <div class="channel-item">
            <div class="channel-info">
                <div class="channel-username"><i class="fab fa-telegram"></i> @${channel.username}</div>
                <div class="channel-title">${channel.title || 'بدون عنوان'}</div>
            </div>
            <div class="channel-actions">
                <button class="action-btn edit" onclick="editChannel('${channel.username}', '${(channel.title || '').replace(/'/g, "\\'")}')">
                    <i class="fas fa-edit"></i>
                </button>
                <button class="action-btn delete" onclick="confirmDeleteChannel('${channel.username}')">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        </div>
    `).join('');
}

function showAddChannelModal() {
    document.getElementById('addChannelModal').style.display = 'flex';
    document.getElementById('newChannelUsername').value = '';
    document.getElementById('newChannelTitle').value = '';
    document.getElementById('channelCheckStatus').innerHTML = '';
    document.getElementById('addChannelBtn').disabled = true;
    document.getElementById('addChannelBtn').style.opacity = '0.5';
}

function closeAddChannelModal() {
    document.getElementById('addChannelModal').style.display = 'none';
}

let channelCheckTimeout;
async function checkChannelInfo() {
    clearTimeout(channelCheckTimeout);
    
    const usernameInput = document.getElementById('newChannelUsername');
    const titleInput = document.getElementById('newChannelTitle');
    const statusDiv = document.getElementById('channelCheckStatus');
    const addBtn = document.getElementById('addChannelBtn');
    const token = localStorage.getItem('admin_token');
    
    const username = usernameInput.value.trim();
    
    // Reset
    titleInput.value = '';
    addBtn.disabled = true;
    addBtn.style.opacity = '0.5';
    
    if (!username || username.length < 2) {
        statusDiv.innerHTML = '';
        return;
    }
    
    // Show loading
    statusDiv.innerHTML = '<i class="fas fa-spinner fa-spin" style="color: #007AFF;"></i> در حال بررسی...';
    
    // Debounce
    channelCheckTimeout = setTimeout(async () => {
        const cleanUsername = username.replace('@', '');
        
        try {
            const response = await fetch(API_URL, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=checkChannelInfo&token=${token}&username=${encodeURIComponent(cleanUsername)}`
            });

            const data = await response.json();
            
            if (data.success) {
                titleInput.value = data.data.title;
                statusDiv.innerHTML = `
                    <div style="color: #34C759; display: flex; align-items: center; gap: 6px;">
                        <i class="fas fa-check-circle"></i>
                        <span>کانال تایید شد - ${data.data.title}</span>
                    </div>
                `;
                addBtn.disabled = false;
                addBtn.style.opacity = '1';
            } else {
                statusDiv.innerHTML = `
                    <div style="color: #FF3B30; display: flex; align-items: center; gap: 6px;">
                        <i class="fas fa-exclamation-circle"></i>
                        <span>${data.message}</span>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error checking channel:', error);
            statusDiv.innerHTML = `
                <div style="color: #FF3B30; display: flex; align-items: center; gap: 6px;">
                    <i class="fas fa-exclamation-circle"></i>
                    <span>خطا در اتصال به سرور</span>
                </div>
            `;
        }
    }, 800);
}

async function addChannel() {
    const username = document.getElementById('newChannelUsername').value.trim();
    const title = document.getElementById('newChannelTitle').value.trim();
    const token = localStorage.getItem('admin_token');
    
    if (!username || !title) {
        showNotification('لطفا ابتدا اطلاعات کانال را تایید کنید', 'error');
        return;
    }
    
    // Remove @ if user entered it
    const cleanUsername = username.replace('@', '');
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=addForceJoinChannel&token=${token}&username=${encodeURIComponent(cleanUsername)}&title=${encodeURIComponent(title)}`
        });

        const data = await response.json();
        
        if (data.success) {
            showNotification('کانال با موفقیت اضافه شد', 'success');
            closeAddChannelModal();
            loadForceJoinChannels();
        } else {
            showNotification(data.message || 'خطا در افزودن کانال', 'error');
        }
    } catch (error) {
        console.error('Error adding channel:', error);
        showNotification('خطا در اتصال به سرور', 'error');
    }
}

function editChannel(username, title) {
    document.getElementById('editChannelModal').style.display = 'flex';
    document.getElementById('editChannelUsername').value = username;
    document.getElementById('editChannelTitle').value = title;
}

function closeEditChannelModal() {
    document.getElementById('editChannelModal').style.display = 'none';
}

async function saveChannelEdits() {
    const username = document.getElementById('editChannelUsername').value.trim();
    const title = document.getElementById('editChannelTitle').value.trim();
    const token = localStorage.getItem('admin_token');
    
    if (!username) {
        showNotification('یوزرنیم کانال نمی‌تواند خالی باشد', 'error');
        return;
    }
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=updateForceJoinChannel&token=${token}&username=${encodeURIComponent(username)}&title=${encodeURIComponent(title)}`
        });

        const data = await response.json();
        
        if (data.success) {
            showNotification('کانال با موفقیت بروزرسانی شد', 'success');
            closeEditChannelModal();
            loadForceJoinChannels();
        } else {
            showNotification(data.message || 'خطا در بروزرسانی کانال', 'error');
        }
    } catch (error) {
        console.error('Error updating channel:', error);
        showNotification('خطا در اتصال به سرور', 'error');
    }
}

function confirmDeleteChannel(username) {
    showConfirmModal(
        'حذف کانال',
        `آیا از حذف کانال @${username} مطمئن هستید؟`,
        () => deleteChannel(username),
        'حذف',
        'danger'
    );
}

async function deleteChannel(username) {
    const token = localStorage.getItem('admin_token');
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=deleteForceJoinChannel&token=${token}&username=${encodeURIComponent(username)}`
        });

        const data = await response.json();
        
        if (data.success) {
            showNotification('کانال با موفقیت حذف شد', 'success');
            loadForceJoinChannels();
        } else {
            showNotification(data.message || 'خطا در حذف کانال', 'error');
        }
    } catch (error) {
        console.error('Error deleting channel:', error);
        showNotification('خطا در اتصال به سرور', 'error');
    }
}

// Webhook Management Functions
async function loadWebhookInfo() {
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'getWebhookInfo',
                token: localStorage.getItem('admin_token')
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            displayWebhookInfo(result.data);
        } else {
            document.getElementById('webhookInfo').innerHTML = `
                <div style="text-align: center; padding: 40px; color: #FF3B30;">
                    <i class="fas fa-exclamation-circle" style="font-size: 32px; margin-bottom: 12px;"></i>
                    <p>خطا در دریافت اطلاعات وب‌هوک</p>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error loading webhook info:', error);
        showNotification('خطا در بارگذاری اطلاعات وب‌هوک', 'error');
    }
}

function displayWebhookInfo(data) {
    const container = document.getElementById('webhookInfo');
    
    if (!data || !data.url) {
        container.innerHTML = `
            <div class="webhook-status" style="padding: 24px; background: rgba(255, 149, 0, 0.1); border-radius: 14px; border-right: 4px solid #FF9500;">
                <h4 style="color: #FF9500; margin-bottom: 8px; display: flex; align-items: center; gap: 8px;">
                    <i class="fas fa-exclamation-triangle"></i>
                    وب‌هوک تنظیم نشده
                </h4>
                <p style="color: #8E8E93; font-size: 14px; margin: 0;">
                    برای فعال شدن ربات، ابتدا وب‌هوک را تنظیم کنید
                </p>
            </div>
        `;
        return;
    }
    
    const statusColor = data.has_custom_certificate ? '#FF9500' : '#34C759';
    const statusText = data.has_custom_certificate ? 'با گواهی سفارشی' : 'فعال';
    
    container.innerHTML = `
        <div class="webhook-status" style="padding: 24px; background: rgba(52, 199, 89, 0.1); border-radius: 14px; border-right: 4px solid ${statusColor};">
            <h4 style="color: ${statusColor}; margin-bottom: 16px; display: flex; align-items: center; gap: 8px;">
                <i class="fas fa-check-circle"></i>
                وب‌هوک ${statusText}
            </h4>
            
            <div style="display: grid; gap: 12px;">
                <div class="info-row" style="display: flex; justify-content: space-between; padding: 12px; background: rgba(0,0,0,0.03); border-radius: 10px;">
                    <span style="color: #8E8E93; font-size: 14px;"><i class="fas fa-link"></i> آدرس:</span>
                    <span style="color: #1C1C1E; font-size: 14px; font-family: monospace; direction: ltr; text-align: left; word-break: break-all;">${data.url}</span>
                </div>
                
                <div class="info-row" style="display: flex; justify-content: space-between; padding: 12px; background: rgba(0,0,0,0.03); border-radius: 10px;">
                    <span style="color: #8E8E93; font-size: 14px;"><i class="fas fa-envelope"></i> آپدیت‌های در انتظار:</span>
                    <span style="color: #1C1C1E; font-size: 14px; font-weight: 600;">${data.pending_update_count || 0}</span>
                </div>
                
                ${data.last_error_date ? `
                <div class="info-row" style="display: flex; flex-direction: column; gap: 8px; padding: 12px; background: rgba(255, 59, 48, 0.1); border-radius: 10px; border: 1px solid rgba(255, 59, 48, 0.2);">
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #FF3B30; font-size: 14px; font-weight: 600;"><i class="fas fa-exclamation-circle"></i> آخرین خطا:</span>
                        <span style="color: #8E8E93; font-size: 12px;">${new Date(data.last_error_date * 1000).toLocaleString('fa-IR')}</span>
                    </div>
                    <p style="color: #FF3B30; font-size: 13px; margin: 0; font-family: monospace; direction: ltr; text-align: left;">${data.last_error_message}</p>
                </div>
                ` : ''}
                
                ${data.ip_address ? `
                <div class="info-row" style="display: flex; justify-content: space-between; padding: 12px; background: rgba(0,0,0,0.03); border-radius: 10px;">
                    <span style="color: #8E8E93; font-size: 14px;"><i class="fas fa-server"></i> IP سرور:</span>
                    <span style="color: #1C1C1E; font-size: 14px; font-family: monospace;">${data.ip_address}</span>
                </div>
                ` : ''}
                
                ${data.max_connections ? `
                <div class="info-row" style="display: flex; justify-content: space-between; padding: 12px; background: rgba(0,0,0,0.03); border-radius: 10px;">
                    <span style="color: #8E8E93; font-size: 14px;"><i class="fas fa-plug"></i> حداکثر اتصالات:</span>
                    <span style="color: #1C1C1E; font-size: 14px; font-weight: 600;">${data.max_connections}</span>
                </div>
                ` : ''}
                
                ${data.allowed_updates && data.allowed_updates.length > 0 ? `
                <div class="info-row" style="display: flex; flex-direction: column; gap: 8px; padding: 12px; background: rgba(0,0,0,0.03); border-radius: 10px;">
                    <span style="color: #8E8E93; font-size: 14px;"><i class="fas fa-filter"></i> آپدیت‌های مجاز:</span>
                    <div style="display: flex; flex-wrap: wrap; gap: 6px;">
                        ${data.allowed_updates.map(u => `<span style="padding: 4px 10px; background: rgba(10, 132, 255, 0.1); color: #0A84FF; border-radius: 8px; font-size: 12px;">${u}</span>`).join('')}
                    </div>
                </div>
                ` : ''}
            </div>
        </div>
    `;
}

async function setWebhook() {
    const customUrl = document.getElementById('webhookUrl').value.trim();
    
    try {
        const response = await fetch(API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'setWebhook',
                token: localStorage.getItem('admin_token'),
                webhook_url: customUrl
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification('وب‌هوک با موفقیت تنظیم شد', 'success');
            setTimeout(() => loadWebhookInfo(), 1000);
        } else {
            showNotification(result.message || 'خطا در تنظیم وب‌هوک', 'error');
        }
    } catch (error) {
        console.error('Error setting webhook:', error);
        showNotification('خطا در تنظیم وب‌هوک', 'error');
    }
}

async function deleteWebhook() {
    showConfirmModal(
        'حذف وب‌هوک',
        'آیا از حذف وب‌هوک اطمینان دارید؟ ربات تا زمان تنظیم مجدد وب‌هوک غیرفعال خواهد بود.',
        async () => {
            try {
                const response = await fetch(API_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'deleteWebhook',
                        token: localStorage.getItem('admin_token')
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('وب‌هوک با موفقیت حذف شد', 'success');
                    setTimeout(() => loadWebhookInfo(), 1000);
                } else {
                    showNotification(result.message || 'خطا در حذف وب‌هوک', 'error');
                }
            } catch (error) {
                console.error('Error deleting webhook:', error);
                showNotification('خطا در حذف وب‌هوک', 'error');
            }
        }
    );
}

function refreshWebhookInfo() {
    loadWebhookInfo();
    showNotification('در حال به‌روزرسانی...', 'info');
}

// Close modals on outside click
document.addEventListener('click', function(e) {
    const userModal = document.getElementById('userModal');
    const addAdminModal = document.getElementById('addAdminModal');
    const confirmModal = document.getElementById('confirmModal');
    const fileDetailsModal = document.getElementById('fileDetailsModal');
    const editFileModal = document.getElementById('editFileModal');
    const warningModal = document.getElementById('warningModal');
    const rejectReasonModal = document.getElementById('rejectReasonModal');
    
    if (e.target === userModal) {
        closeUserModal();
    }
    
    if (e.target === addAdminModal) {
        closeAddAdminModal();
    }
    
    if (e.target === confirmModal) {
        closeConfirmModal();
    }
    
    if (e.target === fileDetailsModal) {
        closeFileDetailsModal();
    }
    
    if (e.target === editFileModal) {
        closeEditFileModal();
    }
    
    if (e.target === warningModal) {
        closeWarningModal();
    }
    
    if (e.target === rejectReasonModal) {
        closeRejectReasonModal();
    }
    
    const addChannelModal = document.getElementById('addChannelModal');
    const editChannelModal = document.getElementById('editChannelModal');
    
    if (e.target === addChannelModal) {
        closeAddChannelModal();
    }
    
    if (e.target === editChannelModal) {
        closeEditChannelModal();
    }
});
